/*
  	#[ Includes :

	The file user.c contains the user definable routines.
	They are: RunMake plus subsidiaries for the make command
	          handlefold for the load command with the fold option.
	          SumStart, SumStep and SumFinish for the sum command
	In the current file we have an example from a
	calculation involving 3 loop QCD diagrams. (lots of them: > 8000)
*/

#define WITHLONGLONG

#include "manage.h"

char *formexecutable = 0;
char formname[] = "form";
char execstring[1024];
char tmpstring[1024];

/*
  	#] Includes : 
  	#[ RunMake :

	This routine is called when the make command decides that a given
	object has to be treated. It is up to the user to decide what kind
	of treatment this will be.
	The parameters are:
		variety:  user specified 0 or positive integer. This way one
		          can deal with different actions. If the user has not
		          specified a variety its value will be -1.
		inargv:   The list of information in the input object.
		outargv:  A pointer to the list of output objects. This list will
		          have to be replaced by a new list once the action is
		          complete.
		num:      The number of the object in the database.
		outvar:   The name of the primary output variable.
	If everything went fine, the return value should be zero.
	The system takes care that the modified output object is written into
	the output database.
*/

int RunDiagram(char **inargv,char ***outargv,LONG num,char *outvar,int child);
int SumDiagram(char **inargv,char ***outargv,LONG num,char *outvar);
int ConvertDiagram(int variety,char **inargv,char ***outargv,LONG num,char *outvar);

int RunMake(int variety,char **inargv,char ***outargv,LONG num,char *outvar,int child)
{
	if ( formexecutable == 0 ) {
		if ( ( formexecutable = getenv("FORM") ) == 0 ) formexecutable = formname;
	}
	switch ( variety ) {
	case 0:	/* The processing of a diagram by MINCER */
		return(RunDiagram(inargv,outargv,num,outvar,child));
	case 1: /* Convert diagram from QGRAF output to MINCER input */
	case 2:
	case 3:
	case 4:
	case 6:
		return(ConvertDiagram(variety,inargv,outargv,num,outvar));
	case 7:	/* The processing of summing a diagram by MINCER */
		return(SumDiagram(inargv,outargv,num,outvar));
	default:
		printf("Variety %d not implemented. Could not do diagram %ld\n"
		,variety,(LONG)(num+1));
		return(-1);
	}
}

/*
 		#[ RunDiagram :
*/

char *nocopy[7] = { "NAME","TOPO","COLOR","DISKSIZE","EXTIME","FLAVOR","VALUE" };
char *analyze(char *outbuf,long outsize,char *name,char *disksize,char *extime);
static char disksize[50],extime[30],namebuf[30];
static char *outputcopy = 0;

int RunDiagram(char **inargv,char ***outargv,LONG num,char *outvar,int child)
{
	char *dia, *topo, *color, *name, *s, *t, **a, *flavor, *subtopo;
	char *outbuf = 0, *output = 0;
	char **newoutargv;
	long outsize;
	int i,ii,j,coloriszero = 0;
	FILE *f;

	if ( outputcopy ) { free(outputcopy); outputcopy = 0; }
	if ( ( dia = GetSubString("DIA",inargv) ) == 0 ) {
		printf(">>>No variable DIA in object %ld\n",(LONG)(num+1));
		return(-1);
	}
	if ( ( topo = GetSubString("TOPO",inargv) ) == 0 ) {
		printf(">>>No variable TOPO in object %ld\n",(LONG)(num+1));
		return(-1);
	}
	subtopo = GetSubString("SUBTOPO",inargv);
	if ( ( color = GetSubString("COLOR",inargv) ) == 0 ) {
		printf(">>>No variable COLOR in object %ld\n",(LONG)(num+1));
		return(-1);
	}
	if ( ( flavor = GetSubString("FLAVOR",inargv) ) == 0 ) {
		printf(">>>No variable FLAVOR in object %ld\n",(LONG)(num+1));
		return(-1);
	}
	if ( *color == '0' || *topo == '0' ) coloriszero = 1;
	if ( ( name = GetSubString("NAME",inargv) ) == 0 ) {
		sprintf(namebuf,"NAME=d%ld",(LONG)(num+1));
		name = namebuf;
	}
	else name -= 5;
	sprintf(disksize,"DISKSIZE=");
	sprintf(extime,"EXTIME=");
	if ( !coloriszero ) {
	  if ( child >= 0 ) {
		if ( ( f = fopen("diagram.h","w") ) == 0 ) {
			printf("Cannot create the file diagram.h for object %ld\n",(LONG)(num+1));
			return(-1);
		}
		if ( child > 0 ) {
			fprintf(f,"#include diagram`CHILD'.h\n");
			fclose(f);
			sprintf(tmpstring,"diagram%d.h",child);
			if ( ( f = fopen(tmpstring,"w") ) == 0 ) {
				printf("Cannot create the file %s for object %ld\n"
					,tmpstring,(LONG)(num+1));
				return(-1);
			}
		}
		fprintf(f,"L %s=%s\n",name+5,dia);
		fprintf(f,"#define TOPO \"%s\"\n",topo);
		if ( subtopo ) {
			fprintf(f,"#define SUBTOPO \"%s\"\n",subtopo);
		}
		fprintf(f,"#define NAME \"%s\"\n",name+5);
		fclose(f);
		if ( subtopo ) {
			printf(">>>FORM is running %s with topology %s(%s)\n",name+5,topo,subtopo);
		}
		else {
			printf(">>>FORM is running %s with topology %s\n",name+5,topo);
		}
		if ( child > 0 ) {
			sprintf(execstring,"%s -d CHILD=%d calcdia > calcdia%d.log"
										,formexecutable,child,child);
		}
		else {
			sprintf(execstring,"%s calcdia > calcdia.log",formexecutable);
		}
		if ( system(execstring) ) {
			printf("Error returned by system\n");
			return(-1);
		}
		if ( child > 0 ) return(0);
	  }
	  if ( child < 0 ) {
		sprintf(tmpstring,"calcdia%d.log",-child);
	  }
	  else {
		sprintf(tmpstring,"calcdia.log");
	  }
	  {
		if ( ( f = fopen(tmpstring,"rb") ) == 0 ) {
			printf("Cannot open the output file %s\n",tmpstring);
			return(-1);
		}
		fseek(f,0,SEEK_END);
		outsize = ftell(f);
		if ( ( outbuf = (char *)mmalloc(sizeof(char)*(outsize+1)
		,"FORM output file") ) == 0 ) {
			fclose(f);
			return(-1);
		}
		fseek(f,0,SEEK_SET);
		if ( mread(f,outbuf,outsize) ) {
			printf("Error while reading the FORM output file\n");
			fclose(f); free(outbuf);
			return(-1);
		}
		outbuf[outsize] = 0;
		fclose(f);
		output = analyze(outbuf,outsize,name+5,disksize+9,extime+7);
		if ( output == 0 ) {
			printf("Error in analysis of FORM output file\n");
			free(outbuf);
			return(-1);
		}
		s = outvar; while ( *s ) s++;
		*--output = '=';
		while ( s > outvar ) *--output = *--s;
	  }
	}
	else {
		if ( child > 0 ) {
			sprintf(tmpstring,"calcdia%d.log",child);
			unlink(tmpstring);
			sprintf(tmpstring,"diagram%d.h",child);
			unlink(tmpstring);
			return(0);
		}
		extime[7] = '0'; extime[8] = '.'; extime[9] = '0'; extime[10] = '0';
		extime[11] = 0;
		disksize[9] = '0'; disksize[10] = 0;
		s = outvar; i = 12; while ( *s ) { s++; i++; }
		if ( ( outputcopy = (char *)mmalloc(sizeof(char)*i,"output") ) == 0 )
			return(-1);
		sprintf(outputcopy,"%s=undefined",outvar);
	}
	i = 0; a = *outargv;
	while ( *a ) { a++; i++; }
	if ( ( newoutargv = (char **)mmalloc(sizeof(char *)*(i+8)
	,"output strings") ) == 0 ) {
		if ( outbuf ) free(outbuf);
		return(-1);
	}
	ii = i;
	i = 0; a = *outargv; newoutargv[i++] = *a++;
	while ( *a ) {
		if ( ii > 1 ) {
			for ( j = 0; j < 7; j++ ) {
				s = *a; t = nocopy[j];
				while ( *t && *s == *t ) { s++; t++; }
				if ( *t == 0 && *s == '=' ) break;
			}
		}
		else j = 7;
		if ( j >= 7 ) newoutargv[i++] = *a;
		a++;
	}
	if ( !coloriszero ) outputcopy = str_dup(output);
	newoutargv[i++] = name;
	newoutargv[i++] = topo-5;
	newoutargv[i++] = outputcopy;
	newoutargv[i++] = color-6;
	newoutargv[i++] = flavor-7;
	newoutargv[i++] = extime;
	newoutargv[i++] = disksize;
	newoutargv[i]   = 0;
	free(*outargv);
	*outargv = newoutargv;
	if ( outbuf ) free(outbuf);
/*
	Now we have one problem left. The buffer in **outargv is a mess.
*/
	ii = 0; i = 0;
	a = *outargv; a++;
	while ( *a ) {
		s = *a++; i++;
		while ( *s ) { s++; i++; }
	}
	outbuf = (char *)mmalloc(sizeof(char)*i,"output");
	a = *outargv; a++; t = outbuf;
	while ( *a ) {
		s = *a++;
		while ( *s ) *t++ = *s++;
		*t++ = 0;
	}
	free(**outargv);
	**outargv = outbuf;
	return(0);
}

/*
 		#] RunDiagram :
 		#[ SumDiagram :
*/

int SumDiagram(char **inargv,char ***outargv,LONG num,char *outvar)
{
	char *dia, *topo, *color, *name, *s, *t, **a, *flavor;
	char *outbuf = 0, *output = 0;
	char **newoutargv;
	long outsize;
	int i,ii,j,coloriszero = 0;
	FILE *f;

	if ( outputcopy ) { free(outputcopy); outputcopy = 0; }
	if ( ( dia = GetSubString("DIA",inargv) ) == 0 ) {
		printf(">>>No variable DIA in object %ld\n",(LONG)(num+1));
		return(-1);
	}
	if ( ( topo = GetSubString("TOPO",inargv) ) == 0 ) {
		printf(">>>No variable TOPO in object %ld\n",(LONG)(num+1));
		return(-1);
	}
	if ( ( color = GetSubString("COLOR",inargv) ) == 0 ) {
		printf(">>>No variable COLOR in object %ld\n",(LONG)(num+1));
		return(-1);
	}
	if ( ( flavor = GetSubString("FLAVOR",inargv) ) == 0 ) {
		printf(">>>No variable FLAVOR in object %ld\n",(LONG)(num+1));
		return(-1);
	}
	if ( *color == '0' || *topo == '0' ) coloriszero = 1;
	if ( ( name = GetSubString("NAME",inargv) ) == 0 ) {
		sprintf(namebuf,"NAME=d%ld",(LONG)(num+1));
		name = namebuf;
	}
	else name -= 5;
	sprintf(disksize,"DISKSIZE=");
	sprintf(extime,"EXTIME=");
	if ( !coloriszero ) {
	if ( ( f = fopen("diagram.h","w") ) == 0 ) {
		printf("Cannot create the file diagram.h for object %ld\n",(LONG)(num+1));
		return(-1);
	}
	fprintf(f,"L %s=%s\n",name+5,dia);
	fprintf(f,"#define TOPO \"%s\"\n",topo);
	fprintf(f,"#define NAME \"%s\"\n",name+5);
	fclose(f);
	printf(">>>FORM is running %s with topology %s\n",name+5,topo);
	sprintf(execstring,"%s sumdia > sumdia.log",formexecutable);
	if ( system(execstring) ) {
		printf("Error returned by system\n");
		return(-1);
	}
	if ( ( f = fopen("sumdia.log","rb") ) == 0 ) {
		printf("Cannot open the output file sumdia.log\n");
		return(-1);
	}
	fseek(f,0,SEEK_END);
	outsize = ftell(f);
	if ( ( outbuf = (char *)mmalloc(sizeof(char)*(outsize+1)
	,"FORM output file") ) == 0 ) {
		fclose(f);
		return(-1);
	}
	fseek(f,0,SEEK_SET);
	if ( mread(f,outbuf,outsize) ) {
		printf("Error while reading the FORM output file\n");
		fclose(f); free(outbuf);
		return(-1);
	}
	outbuf[outsize] = 0;
	fclose(f);
	output = analyze(outbuf,outsize,name+5,disksize+9,extime+7);
	if ( output == 0 ) {
		printf("Error in analysis of FORM output file\n");
		free(outbuf);
		return(-1);
	}
	s = outvar; while ( *s ) s++;
	*--output = '=';
	while ( s > outvar ) *--output = *--s;
	}
	else {
		extime[7] = '0'; extime[8] = '.'; extime[9] = '0'; extime[10] = '0';
		extime[11] = 0;
		disksize[9] = '0'; disksize[10] = 0;
		s = outvar; i = 12; while ( *s ) { s++; i++; }
		if ( ( outputcopy = (char *)mmalloc(sizeof(char)*i,"output") ) == 0 )
			return(-1);
		sprintf(outputcopy,"%s=undefined",outvar);
	}
	i = 0; a = *outargv;
	while ( *a ) { a++; i++; }
	if ( ( newoutargv = (char **)mmalloc(sizeof(char *)*(i+8)
	,"output strings") ) == 0 ) {
		if ( outbuf ) free(outbuf);
		return(-1);
	}
	ii = i;
	i = 0; a = *outargv; newoutargv[i++] = *a++;
	while ( *a ) {
		if ( ii > 1 ) {
			for ( j = 0; j < 7; j++ ) {
				s = *a; t = nocopy[j];
				while ( *t && *s == *t ) { s++; t++; }
				if ( *t == 0 && *s == '=' ) break;
			}
		}
		else j = 7;
		if ( j >= 7 ) newoutargv[i++] = *a;
		a++;
	}
	if ( !coloriszero ) outputcopy = str_dup(output);
	newoutargv[i++] = name;
	newoutargv[i++] = topo-5;
	newoutargv[i++] = outputcopy;
	newoutargv[i++] = color-6;
	newoutargv[i++] = flavor-7;
	newoutargv[i++] = extime;
	newoutargv[i++] = disksize;
	newoutargv[i]   = 0;
	free(*outargv);
	*outargv = newoutargv;
	if ( outbuf ) free(outbuf);
/*
	Now we have one problem left. The buffer in **outargv is a mess.
*/
	ii = 0; i = 0;
	a = *outargv; a++;
	while ( *a ) {
		s = *a++; i++;
		while ( *s ) { s++; i++; }
	}
	outbuf = (char *)mmalloc(sizeof(char)*i,"output");
	a = *outargv; a++; t = outbuf;
	while ( *a ) {
		s = *a++;
		while ( *s ) *t++ = *s++;
		*t++ = 0;
	}
	free(**outargv);
	**outargv = outbuf;
	return(0);
}

/*
 		#] SumDiagram : 
 		#[ analyze :
*/

char *analyze(char *outbuf,long outsize,char *name,char *disksize,char *extime)
{
	char *s, *t, *u;
#ifdef WITHLONGLONG
	off_t maxsize = 0, size1 = 0, size2 = 0 ,size3 = 0;
	int j = 0, jj = 0;
	char x;
#else
	unsigned long maxsize = 0, size1 = 0, size2 = 0 ,size3 = 0;
#endif
	long n;
/*
	First find the last line that starts with Time;
*/
	s = outbuf + outsize;
	while ( s > outbuf ) {
		if ( *s != 'T' ) { s--; continue; }
		if ( s[1] == 'i' && s[2] == 'm' && s[3] == 'e' && s[4] == ' '
		&& s[5] == '=' ) break;
		s -= 6;
	}
	if ( s <= outbuf ) {
		printf("No timing information\n");
		return(0);
	}
	s += 6;
	while ( *s == ' ' ) s++;
	t = extime;
	while ( *s >= '0' && *s <= '9' ) *t++ = *s++;
	*t++ = *s++; *t++ = *s++; *t++ = *s++; *t = 0;

	s = outbuf; n = outsize;
	while ( n >= 66 ) {
		if ( *s != 'o' ) { s++; n--; continue; }
		if ( s[1] != 'u' || s[2] != 't' || s[3] != 'p' || s[4] != 'u'
		|| s[5] != 't' || s[6] != ' ' || s[7] != '=' ) {
			s += 8; n -= 8; continue;
		}
		size2 = 0;
		if ( s[-166] == 'a' && s[-165] == 'c' && s[-164] == 't'
		&& s[-163] == 'i' && s[-162] == 'v' && s[-161] == 'e' ) {
			t = s - 100;
			while ( *t == ' ' ) t++;
			while ( *t >= '0' && *t <= '9' ) size2 = 10* size2 + *t++ - '0';
		}
		size3 = 0;
		t = s + 62;
		while ( *t == ' ' ) t++;
		while ( *t >= '0' && *t <= '9' ) size3 = 10* size3 + *t++ - '0';
/*		if ( size1+size2+size3 > maxsize ) maxsize = size1+size2+size3; */
		if ( size1 > size3 ) {
			if ( size2+size1 > maxsize ) maxsize = size2+size1;
		}
		else {
			if ( size2+size3 > maxsize ) maxsize = size2+size3;
		}
		size1 = size3;
		s += 150; n -= 150;
	}
#ifdef WITHLONGLONG
	while ( maxsize) {
		x = maxsize % 10 + '0';
		disksize[j] = x; j++;
		maxsize = maxsize/10;
	}
	j--;
	while ( j > jj ) {
		x = disksize[j]; disksize[j] = disksize[jj]; disksize[jj] = x;
		j--; jj++;
	}
#else
	sprintf(disksize,"%ld",maxsize);
#endif

	s = outbuf + outsize;
	while ( s > outbuf ) {
		if ( *s != *name ) { s--; continue; }
		t = s; u = name;
		while ( *u && *t == *u ) { t++; u++; }
		if ( *u == 0 && *t == ' ' && t[1] == '=' ) {
			t += 2;
			s = t;
			u = t;
			while ( *s && *s != ';' ) {
				if ( *s == '\n' || *s == '\r' || *s == ' ' || *s == '\\' )
					{ s++; continue; }
				*t++ = *s++;
			}
			*t = 0;
			return(u);
		}
		s--;
	}
	printf("Cannot find output with name %s in FORM output file\n",name);
	return(0);
}

/*
 		#] analyze : 
 		#[ ConvertDiagram :
*/

char *smaak = "FLAVOR=1";

int ConvertDiagram(int variety,char **inargv,char ***outargv,LONG num,char *outvar)
{
	char *dia, *topo = 0, *color = 0, *flavor = 0, *name, *sf, *s, *t, *u;
	char *outbuf = 0, **a;
	char **newoutargv;
	long outsize;
	int i, ii;
	FILE *f;

	if ( ( dia = GetSubString("GRAF",inargv) ) == 0 ) {
		printf(">>>No variable GRAF in object %ld\n",(LONG)(num+1));
		return(-1);
	}
	if ( ( sf = GetSubString("SF",inargv) ) == 0 ) {
/*
		printf(">>>No variable SF in object %ld\n",(LONG)(num+1));
		return(-1);
*/
	}
	if ( ( name = GetSubString("NAME",inargv) ) == 0 ) {
		sprintf(namebuf,"NAME=d%ld",(LONG)(num+1));
		name = namebuf;
	}
	else name -= 5;

	if ( ( f = fopen("convert.h","w") ) == 0 ) {
		printf("Cannot create the file convert.h for diagram%ld\n",(LONG)(num+1));
		return(-1);
	}
	fprintf(f,"  %s\n",dia);
	if ( sf ) fprintf(f,"#define SF \"%s\"\n",sf);
	fclose(f);
	printf(">>>FORM is converting %s\n",name+5);
	sprintf(execstring,"%s -q -d PROCESS=%d convdia > convdia.log",formexecutable,variety);
	if ( system(execstring) ) {
		printf("Error returned by system\n");
		return(-1);
	}
	if ( ( f = fopen("convdia.log","rb") ) == 0 ) {
		printf("Cannot open the output file convdia.log\n");
		return(-1);
	}
	fseek(f,0,SEEK_END);
	outsize = ftell(f);
	if ( ( outbuf = (char *)mmalloc(sizeof(char)*(outsize+1)
	,"FORM output file") ) == 0 ) {
		fclose(f);
		return(-1);
	}
	fseek(f,0,SEEK_SET);
	if ( mread(f,outbuf,outsize) ) {
		printf("Error while reading the FORM output file convdia.log\n");
		fclose(f); free(outbuf);
		return(-1);
	}
	outbuf[outsize] = 0;
	fclose(f);
/*
	Now we find the pointers to the output
*/
	s = outbuf + outsize;
	while ( s > outbuf ) {
		if ( s[-1] != 'c' ) { s--; continue; }
		t = "colour";
		u = s-1;
		while ( *t && *t == *u ) { t++; u++; }
		if ( *t == 0 && *u == ' ' && u[1] == '=' ) {
			t = "COLOR"; s++;
			color = s;
			while ( *t ) *s++ = *t++;
			s++; t = s;
			while ( *s && *s != ';' ) {
				if ( *s != '\n' && *s != '\r' && *s != ' ' ) *t++ = *s++;
				else s++;
			}
			*t = 0;
			break;
		}
		else s--;
	}
	if ( s == outbuf ) {
		printf("Cannot find colour in the FORM output file convdia.log\n");
		free(outbuf);
		return(-1);
	}
	s = outbuf + outsize;
	while ( s > outbuf ) {
		if ( s[-1] != 't' ) { s--; continue; }
		t = "topology";
		u = s-1;
		while ( *t && *t == *u ) { t++; u++; }
		if ( *t == 0 && *u == ' ' && u[1] == '=' ) {
			t = "TOPO"; s += 4;
			topo = s;
			while ( *t ) *s++ = *t++;
			s++; t = s;
			while ( *s && *s != ';' ) {
				if ( *s != '\n' && *s != '\r' && *s != ' ' ) *t++ = *s++;
				else s++;
			}
			*t = 0;
			break;
		}
		else s--;
	}
	if ( s == outbuf ) {
		printf("Cannot find topology in the FORM output file convdia.log\n");
		free(outbuf);
		return(-1);
	}
	s = outbuf + outsize;
	while ( s > outbuf ) {
		if ( s[-1] != 'e' ) { s--; continue; }
		t = "expr1";
		u = s-1;
		while ( *t && *t == *u ) { t++; u++; }
		if ( *t == 0 && *u == ' ' && u[1] == '=' ) {
			if ( outvar == 0 ) {
				t = "DIA"; s = u-2;
				outputcopy = s;
				while ( *t ) *s++ = *t++;
				s++;
			}
			else { s = u+2; outputcopy = s; }
			t = s;
			while ( *s && *s != ';' ) {
				if ( *s != '\n' && *s != '\r' && *s != ' ' ) *t++ = *s++;
				else s++;
			}
			*t++ = ';'; *t = 0;
			break;
		}
		else s--;
	}
	if ( s == outbuf ) {
		printf("Cannot find the result in the FORM output file convdia.log\n");
		free(outbuf);
		return(-1);
	}
	s = outbuf + outsize;
	while ( s > outbuf ) {
		if ( s[-1] != 'f' ) { s--; continue; }
		t = "flavour";
		u = s-1;
		while ( *t && *t == *u ) { t++; u++; }
		if ( *t == 0 && *u == ' ' && u[1] == '=' ) {
			t = "FLAVOR"; s++;
			flavor = s;
			while ( *t ) *s++ = *t++;
			s++; t = s;
			while ( *s && *s != ';' ) {
				if ( *s != '\n' && *s != '\r' && *s != ' ' ) *t++ = *s++;
				else s++;
			}
			*t = 0;
			break;
		}
		else s--;
	}
	if ( s == outbuf ) {
		if ( variety < 3 ) {
			printf("Cannot find flavour in the FORM output file convdia.log\n");
			free(outbuf);
			return(-1);
		}
		else flavor = smaak;
	}
/*
	Having found the pointers we put them in the output and finish
*/
	s = outvar; while ( *s ) s++;
	*--outputcopy = '=';
	while ( s > outvar ) *--outputcopy = *--s;
	if ( ( newoutargv = (char **)mmalloc(sizeof(char *)*7
	,"output strings") ) == 0 ) {
		if ( outbuf ) free(outbuf);
		return(-1);
	}
	i = 0;
	newoutargv[i++] = outbuf;
	newoutargv[i++] = name;
	newoutargv[i++] = outputcopy;
	newoutargv[i++] = topo;
	newoutargv[i++] = color;
	newoutargv[i++] = flavor;
	newoutargv[i]   = 0;
	if ( *outargv ) free(*outargv);
	*outargv = newoutargv;
/*
	Now we have one problem left. The buffer in **outargv is a mess.
*/
	ii = 0; i = 0;
	a = *outargv; a++;
	while ( *a ) {
		s = *a++; i++;
		while ( *s ) { s++; i++; }
	}
	outbuf = (char *)mmalloc(sizeof(char)*i,"output");
	a = *outargv; a++; t = outbuf;
	while ( *a ) {
		s = *a++;
		while ( *s ) *t++ = *s++;
		*t++ = 0;
	}
	free(**outargv);
	**outargv = outbuf;
	return(0);
}

/*
 		#] ConvertDiagram : 
  	#] RunMake :
  	#[ handlefold :

	Used for reading data in a file with folds into a database d.
	The contents of the fold from start to finish.
	The name of the fold is in 'name'. It is zero terminated.
*/

int handlefold(DBASE *d,char *start,char *finish,char *name,char *varname)
{
	LONG size;
	char **argv, *buffer, *s, *t, *u, c;
	int num = 3;
	if ( *name != 'd' ) return(0);
	s = start;
	while ( s < finish ) {
		if ( ( *s == '\n' || *s == '\r' )  && s[1] == '#' && s+1 < finish ) num++;
		s++;
	}
	size = finish - start + 8;
	s = varname;
	while ( *s ) { size++; s++; }
	s = name;
	while ( *s ) { size++; s++; }
	if ( ( buffer = (char *)mmalloc(sizeof(char)*size,"fold buffer") ) == 0 )
		return(-1);
	if ( ( argv = (char **)mmalloc(sizeof(char *)*(num+1),"new object data") ) == 0 )
		{ free(buffer); return(-1); }
	argv[0] = buffer;
	argv[1] = buffer;
	num = 2;
	t = buffer;
	s = "NAME="; while ( *s ) *t++ = *s++;
	s = name; while ( *s ) *t++ = *s++; *t++ = 0;
	argv[num++] = t;
	s = varname;
	while ( *s ) *t++ = *s++;
	*t++ = '=';
	s = start;
	while ( s < finish ) {
		if ( *s == '*' ) {
			while ( s < finish && *s != '\n' && *s != '\r' ) s++;
			s++;
		}
		else if ( *s == '#' ) {	/* #define VAR "value" */
			*t++ = 0;
			argv[num++] = t;
			s++;
			u = s + 6;
			c = *u; *u = 0;
			if ( strcmp(s,"define") ) {
				printf("Unexpected # instruction in input fold %s\n",name);
getout:
				free(buffer); free(argv);
				return(-1);
			}
			*u = c; s += 6;
			while ( ( *s == ' ' || *s == '\t' ) && s < finish ) s++;
			while ( s < finish && *s != ' ' && *s != '\t' && *s != '"' )
				*t++ = *s++;
			*t++ = '=';
			while ( ( *s == ' ' || *s == '\t' ) && s < finish ) s++;
			if ( *s != '"' ) {
IllDef:
				printf("Illegal define statement in input fold %s\n",name);
				goto getout;
			}
			s++;
			while ( s < finish && *s != '"' ) { *t++ = *s++; }
			if ( *s != '"' ) goto IllDef;
			s++;
			*t++ = 0;
			while ( s < finish && *s != '\n' && *s != '\r' ) s++;
			s++;
		}
		else if ( *s == '\n' || *s == '\r' ) s++;
		else {
			while ( s < finish && ( *s == ' ' || *s == '\t' ) ) s++;
			while ( s < finish && *s != '\n' && *s != '\r' ) *t++ = *s++;
			s++;
		}
	}
	*t = 0; argv[num] = 0;
	size = AddObject(d,argv);
	free(buffer); free(argv);
	if ( size < 0 ) return(-1);
	return(0);
}

/*
  	#] handlefold : 
  	#[ Sum :
 		#[ Variables :
*/

static LONG numberofsteps;
static FILE *SumFile;
static long sumtime;
static long diskspace;

/*
 		#] Variables : 
 		#[ SumStart :
*/

int SumStart(int variety,char *outname)
{
	int error = 0;
	numberofsteps = 0;
	switch ( variety ) {
	case 0:		/* adds VALUE weighted with COLOR using FORM */
	case 3:		/* adds VALUE not weighted using FORM */
	case 4:		/* adds VALUE weighted with COLOR using FORM without missing */
	case 5:		/* adds VALUE weighted with COLOR and FLAVOR using FORM */
		if ( ( SumFile = fopen("totalinp.h","wb") ) == 0 ) {
			printf("Cannot create totalinp.h\n");
			return(-1);
		}
		fputs("#-\nS undefined;\n.global\nL Fsum =\n",SumFile);
		break;
	case 1:		/* adds EXTIME */
		sumtime = 0;
		break;
	case 2:		/* Determines maximum diskspace */
		diskspace = 0;
		break;
	default:
		printf("The variety %d has not been implemented for the sum command\n"
			,variety);
		error = -1;
		break;
	}
	return(error);
}

/*
 		#] SumStart : 
 		#[ SumStep :
*/

int SumStep(int variety,char **argv,LONG num,char *outname)
{
	int error = 0;
	char *s, *color, *flavor;
	switch ( variety ) {
	case 0:
		if ( ( s = GetSubString("VALUE",argv) ) == 0
		|| ( color = GetSubString("COLOR",argv) ) == 0 ) {
			printf("No complete results in object %ld\n",(long)(num+1));
			return(-1);
		}
		if ( numberofsteps % 50 == 0 && numberofsteps != 0 ) {
			fputs(" ;\n.sort\nL Fsum = Fsum\n",SumFile);
		}
		numberofsteps++;
		fprintf(SumFile," +(%s)*(%s)\n",color,s);
		break;
	case 1:
		if ( ( s = GetSubString("EXTIME",argv) ) != 0 ) {
			long sectime; int hundreth;
			sectime = 0;
			while ( *s >= '0' && *s <= '9' ) sectime = 10*sectime + *s++ - '0';
			if ( *s && *s != '.' ) {
ervalEX:		printf(" Error in value of EXTIME in object %ld\n",
					(long)(num+1));
				return(-1);
			}
			if ( *s ) s++;
			hundreth = 0;
			while ( *s >= '0' && *s <= '9' ) hundreth = 10*hundreth + *s++ - '0';
			if ( *s ) goto ervalEX;
			sumtime += 100*sectime + hundreth;
		}
		break;
	case 2:
		if ( ( s = GetSubString("DISKSIZE",argv) ) != 0 ) {
			long disk;
			disk = 0;
			while ( *s >= '0' && *s <= '9' ) disk = 10*disk + *s++ - '0';
			if ( *s ) {
				printf(" Error in value of DISKSIZE in object %ld\n",
					(long)(num+1));
				return(-1);
			}
			if ( diskspace < disk ) diskspace = disk;
		}
		break;
	case 3:
		if ( ( s = GetSubString("VALUE",argv) ) == 0 ) {
			printf("No complete results in object %ld\n",(long)(num+1));
			return(-1);
		}
		if ( numberofsteps % 50 == 0 && numberofsteps != 0 ) {
			fputs(" ;\n.sort\nL Fsum = Fsum\n",SumFile);
		}
		numberofsteps++;
		fprintf(SumFile," +(%s)\n",s);
		break;
	case 4:
		if ( ( s = GetSubString("VALUE",argv) ) == 0
		|| ( color = GetSubString("COLOR",argv) ) == 0 ) {}
		else {
			if ( numberofsteps % 50 == 0 && numberofsteps != 0 ) {
				fputs(" ;\n.sort\nL Fsum = Fsum\n",SumFile);
			}
			numberofsteps++;
			fprintf(SumFile," +(%s)*(%s)\n",color,s);
		}
		break;
	case 5:
		if ( ( s = GetSubString("VALUE",argv) ) == 0
		|| ( color = GetSubString("COLOR",argv) ) == 0
		|| ( flavor = GetSubString("FLAVOR",argv) ) == 0 ) {
			printf("No complete results in object %ld\n",(long)(num+1));
			return(-1);
		}
		if ( numberofsteps % 50 == 0 && numberofsteps != 0 ) {
			fputs(" ;\n.sort\nL Fsum = Fsum\n",SumFile);
		}
		numberofsteps++;
		fprintf(SumFile," +(%s)*(%s)*(%s)\n",flavor,color,s);
		break;
	default:
		printf("The variety %d has not been implemented for the sum command\n"
			,variety);
		error = -1;
		break;
	}
	return(error);
}

/*
 		#] SumStep : 
 		#[ SpecStep :
*/

int SpecStep(int variety,char **argv1,char **argv2,LONG num,char *outname)
{
	int error = 0;
	char *s, *color, *flavor;
	switch ( variety ) {
	case 0:
		if ( ( s = GetSubString("VALUE",argv2) ) == 0
		|| ( color = GetSubString("COLOR",argv1) ) == 0 ) {
			printf("No complete results in object %ld\n",(long)(num+1));
			return(-1);
		}
		if ( numberofsteps % 50 == 0 && numberofsteps != 0 ) {
			fputs(" ;\n.sort\nL Fsum = Fsum\n",SumFile);
		}
		numberofsteps++;
		fprintf(SumFile," +(%s)*(%s)\n",color,s);
		break;
	case 3:
		if ( ( s = GetSubString("VALUE",argv2) ) == 0 ) {
			printf("No complete results in object %ld\n",(long)(num+1));
			return(-1);
		}
		if ( numberofsteps % 50 == 0 && numberofsteps != 0 ) {
			fputs(" ;\n.sort\nL Fsum = Fsum\n",SumFile);
		}
		numberofsteps++;
		fprintf(SumFile," +(%s)\n",s);
		break;
	case 4:
		if ( ( s = GetSubString("VALUE",argv2) ) == 0
		|| ( color = GetSubString("COLOR",argv1) ) == 0 ) {}
		else {
			if ( numberofsteps % 50 == 0 && numberofsteps != 0 ) {
				fputs(" ;\n.sort\nL Fsum = Fsum\n",SumFile);
			}
			numberofsteps++;
			fprintf(SumFile," +(%s)*(%s)\n",color,s);
		}
		break;
	case 5:
		if ( ( s = GetSubString("VALUE",argv2) ) == 0
		|| ( color = GetSubString("COLOR",argv1) ) == 0
		|| ( flavor = GetSubString("FLAVOR",argv1) ) == 0 ) {
			printf("No complete results in object %ld\n",(long)(num+1));
			return(-1);
		}
		if ( numberofsteps % 50 == 0 && numberofsteps != 0 ) {
			fputs(" ;\n.sort\nL Fsum = Fsum\n",SumFile);
		}
		numberofsteps++;
		fprintf(SumFile," +(%s)*(%s)*(%s)\n",flavor,color,s);
		break;
	default:
		printf("The variety %d has not been implemented for the special command\n"
			,variety);
		error = -1;
		break;
	}
	return(error);
}

/*
 		#] SpecStep : 
 		#[ SumFinish :
*/

int SumFinish(int variety,char *outname)
{
	int error = 0;
	if ( formexecutable == 0 ) {
		if ( ( formexecutable = getenv("FORM") ) == 0 ) formexecutable = formname;
	}
	switch ( variety ) {
	case 0:
	case 3:
	case 4:
	case 5:
		fputs(" ;\n",SumFile);
		fclose(SumFile);
		if ( outname ) sprintf(execstring,"%s total > %s",formexecutable,outname);
		else           sprintf(execstring,"%s total > total.log",formexecutable);
		printf("system %s\n",execstring);
		if ( ( error = system(execstring) ) != 0 ) {
			printf("system returned with error code %d\n",error);
		}
		else {
			if ( outname )
				printf("The output can be found in the file %s\n",outname);
			else
				printf("The output can be found in the file total.log\n");
		}
		break;
	case 1:
		printf("Total execution time is %ld.%2d sec\n",sumtime/100,
			(int)(sumtime%100));
		break;
	case 2:
		printf("Maximum disk space used is %ld bytes\n",diskspace);
		break;
	default:
		printf("The variety %d has not been implemented for the sum command\n"
			,variety);
		error = -1;
		break;
	}
	return(error);
}

/*
 		#] SumFinish : 
  	#] Sum : 
  	#[ LinSum :
 		#[ Variables :
*/

/*
 		#] Variables : 
 		#[ LinSumStart :
*/

int LinSumStart(int variety,char *outname)
{
	int error = 0;
	numberofsteps = 0;
	switch ( variety ) {
	case -1:
/*
		Opens the output file
*/
		if ( ( SumFile = fopen("lintot.h","wb") ) == 0 ) {
			printf("Cannot create lintot.h\n");
			return(-1);
		}
		fputs("#-\nS D;\n",SumFile);
		break;
	case 0:
/*
		adds VALUE weighted with COLOR and a symbol for the diagram,
		using FORM
*/
		fputs("S n,[n-4],ep(:6);\ndimension n:[n-4];\nS z3,z4,z5;\n",SumFile);
		fputs("S xi,cw1,cw2,cw3,cf,ca,nf,nc,[cf-ca/2],[cf-ca],[dabc^2/n];\n",SumFile);
		fputs("S undefined;\n.global\nL Fsum =\n",SumFile);
		break;
	default:
		printf("The variety %d has not been implemented for the linsum command\n"
			,variety);
		error = -1;
		break;
	}
	return(error);
}

/*
 		#] LinSumStart : 
 		#[ LinSumStep :
*/

int LinSumStep(int variety,char **argv,LONG num,char *outname)
{
	int error = 0, i;
	char *s, *color, *t, c;
	switch ( variety ) {
	case 0:
		if ( ( s = GetSubString("VALUE",argv) ) == 0
		|| ( color = GetSubString("COLOR",argv) ) == 0 ) {
			printf("No complete results in object %ld\n",(long)(num+1));
			return(-1);
		}
		if ( numberofsteps % 50 == 0 && numberofsteps != 0 ) {
			fputs(" ;\n.sort\nL Fsum = Fsum\n",SumFile);
		}
		numberofsteps++;
		fprintf(SumFile," +D^%ld*(%s)*(",num,color);
		t = s;
		for (;;) {
			i = 0;
			while ( *t && i < 71 ) { t++; i++; }
			if ( *t == 0 ) {
				fprintf(SumFile,"%s)\n",s);
				break;
			}
			c = *t; *t = 0;
			fprintf(SumFile,"%s\\\n",s);
			*t = c; s = t;
		}
		break;
	default:
		printf("The variety %d has not been implemented for the linsum command\n"
			,variety);
		error = -1;
		break;
	}
	return(error);
}

/*
 		#] LinSumStep : 
 		#[ LinSumDecl :
*/

int LinSumDecl(int variety,char **argv,LONG num,char *outname)
{
	int error = 0;
	char *s, *color;
	switch ( variety ) {
	case 0:
		if ( ( s = GetSubString("VALUE",argv) ) == 0
		|| ( color = GetSubString("COLOR",argv) ) == 0 ) {
			printf("No complete results in object %ld\n",(long)(num+1));
			return(-1);
		}
		fprintf(SumFile,"S D%ld;\n",num);
		break;
	default:
		printf("The variety %d has not been implemented for the linsum command\n"
			,variety);
		error = -1;
		break;
	}
	return(error);
}

/*
 		#] LinSumDecl : 
 		#[ LinSumFinish :
*/

int LinSumFinish(int variety,char *outname)
{
	int error = 0;
	if ( formexecutable == 0 ) {
		if ( ( formexecutable = getenv("FORM") ) == 0 ) formexecutable = formname;
	}
	switch ( variety ) {
	case 0:
		fputs(" ;\n",SumFile);
		fclose(SumFile);
/*
		if ( outname ) sprintf(execstring,"%s total > %s",formexecutable,outname);
		else           sprintf(execstring,"%s total > total.log",formexecutable);
		printf("system %s\n",execstring);
		if ( ( error = system(execstring) ) != 0 ) {
			printf("system returned with error code %d\n",error);
		}
		else {
			if ( outname )
				printf("The output can be found in the file %s\n",outname);
			else
				printf("The output can be found in the file total.log\n");
		}
*/
		printf("Declarations and expression in lintot.h\n");
		break;
	default:
		printf("The variety %d has not been implemented for the linsum command\n"
			,variety);
		error = -1;
		break;
	}
	return(error);
}

/*
 		#] LinSumFinish : 
  	#] LinSum : 
*/
