/*
  	#[ Includes :

	File contains the lowlevel routines for the management of primitive
	database-like files. The structures are explained in the file manage.h
	File made by J.Vermaseren, april-1994.
*/

#include "manage.h"

/*
  	#] Includes :
  	#[ Variables :
*/
 
static INDEXBLOCK scratchblock;
 
#define CFD(y,s,type,x,j) for(x=0,j=0;j<sizeof(type);j++) \
	x=(x<<8)+((*s++)&0x00FF); y=x;
#define CTD(y,s,type,x,j) x=y;for(j=sizeof(type)-1;j>=0;j--){s[j]=x&0xFF; \
                      x>>=8;} s += sizeof(type);

/*
  	#] Variables :
  	#[ Utilities :
 		#[ mmalloc :
*/

void *mmalloc(size_t size,char *message)
{
	void *m;
	m = malloc(size);
	if ( m == 0 ) printf("No memory for allocation of %s\n",message);
	return(m);
}

/*
 		#] mmalloc :
 		#[ mread :
*/

int mread(FILE *f,char *buffer,LONG size)
{
	LONG x;
	while ( size > 0 ) {
		x = fread(buffer,sizeof(char),size,f);
		if ( x <= 0 ) return(-1);
		buffer += x;
		size -= x;
	}
	return(0);
}

/*
 		#] mread :
 		#[ mwrite :
*/

int mwrite(FILE *f,char *buffer,LONG size)
{
	LONG x;
	while ( size > 0 ) {
		x = fwrite(buffer,sizeof(char),size,f);
		if ( x <= 0 ) return(-1);
		buffer += x;
		size -= x;
	}
	if ( withoutflush == 0 ) fflush(f);
	return(0);
}

/*
 		#] mwrite :
 		#[ str_dup :
*/

char *str_dup(char *str)
{
	char *s, *t;
	int i;
	s = str;
	while ( *s ) s++;
	i = s - str + 1;
	if ( ( s = mmalloc((size_t)i,"a string copy") ) == 0 ) return(0);
	t = s;
	while ( *str ) *t++ = *str++;
	*t = 0;
	return(s);
}

/*
 		#] str_dup :
 		#[ convertblock :
*/

void convertblock(INDEXBLOCK *in,INDEXBLOCK *out,int mode)
{
	char *s;
	LONG i, x;
	int j;
	OBJECTS *obj;
	switch ( mode ) {
	case TODISK:
		s = (char *)out;
		CTD(in->flags,s,LONG,x,j)
		CTD(in->previousblock,s,LONG,x,j)
		CTD(in->position,s,LONG,x,j)
		for ( i = 0, obj = in->objects; i < NUMOBJECTS; i++, obj++ ) {
			CTD(obj->position,s,LONG,x,j)
			CTD(obj->size,s,LONG,x,j)
			CTD(obj->date,s,time_t,x,j)
		}
		break;
	case FROMDISK:
		s = (char *)in;
		CFD(out->flags,s,LONG,x,j)
		CFD(out->previousblock,s,LONG,x,j)
		CFD(out->position,s,LONG,x,j)
		for ( i = 0, obj = out->objects; i < NUMOBJECTS; i++, obj++ ) {
			CFD(obj->position,s,LONG,x,j)
			CFD(obj->size,s,LONG,x,j)
			CFD(obj->date,s,time_t,x,j)
		}
		break;
	}
}

/*
 		#] convertblock :
 		#[ convertiniinfo :
*/

void convertiniinfo(INIINFO *in,INIINFO *out,int mode)
{
	char *s;
	LONG i, x, *y;
	int j;
	switch ( mode ) {
	case TODISK:
		s = (char *)out; y = (LONG *)in;
		for ( i = sizeof(INIINFO)/sizeof(LONG); i > 0; i-- ) {
			CTD(*y,s,LONG,x,j)
			y++;
		}
		break;
	case FROMDISK:
		s = (char *)in; y = (LONG *)out;
		for ( i = sizeof(INIINFO)/sizeof(LONG); i > 0; i-- ) {
			CFD(*y,s,LONG,x,j)
			y++;
		}
		break;
	}
}

/*
 		#] convertiniinfo :
  	#] Utilities :
  	#[ ReadIndex :
*/

int ReadIndex(DBASE *d)
{
	LONG i;
	INDEXBLOCK **ib;
	LONG position, size;
/*
	Allocate the pieces one by one (makes it easier to give it back)
*/
	if ( d->info.numberofindexblocks <= 0 ) return(0);
	size = sizeof(INDEXBLOCK *)*d->info.numberofindexblocks;
	if ( size > 16000000 ) {
		printf("We need more than 16 Mbytes for the index.\n");
		printf("The file %s may not be a proper database\n",d->name);
		return(-1);
	}
	if ( ( ib = mmalloc(size,
	"index") ) == 0 ) return(-1);
	for ( i = 0; i < d->info.numberofindexblocks; i++ ) {
		if ( ( ib[i] = mmalloc(sizeof(INDEXBLOCK),"index block") ) == 0 ) {
			for ( --i; i >= 0; i-- ) free(ib[i]);
			free(ib);
			return(-1);
		}
	}
/*
	Read the index blocks, from the back to the front. The links are only
	reliable that way.
*/
	position = d->info.lastindexblock;
	for ( i = d->info.numberofindexblocks - 1; i >= 0; i-- ) {
		fseek(d->handle,position,SEEK_SET);
		if ( mread(d->handle,(char *)(&scratchblock),sizeof(INDEXBLOCK)) ) {
			printf("Error while reading file %s\n",d->name);
thisiswrong:
			for ( i = 0; i < d->info.numberofindexblocks; i++ ) free(ib[i]);
			free(ib);
			return(-1);
		}
		convertblock(&scratchblock,ib[i],FROMDISK);
		if ( ib[i]->position != position ||
		( ib[i]->previousblock <= 0 && i > 0 ) ) {
			printf("File %s has inconsistent contents\n",d->name);
			goto thisiswrong;
		}
		position = ib[i]->previousblock;
	}
	d->info.firstindexblock = ib[0]->position;
	for ( i = 0; i < d->info.numberofindexblocks; i++ ) {
		ib[i]->flags &= CLEANFLAG;
	}
/*
	Give the old info back to the system.
*/
	if ( d->iblocks ) {
		for ( i = 0; i < d->info.numberofindexblocks; i++ ) {
			if ( d->iblocks[i] ) free(d->iblocks[i]);
		}
		free(d->iblocks);
	}
/*
	And substitute the new blocks
*/
	d->iblocks = ib;
	return(0);
}

/*
  	#] ReadIndex :
  	#[ WriteIndexBlock :
*/

int WriteIndexBlock(DBASE *d,LONG num)
{
	if ( num >= d->info.numberofindexblocks ) {
		printf("Illegal number specified for number of index blocks\n");
		return(-1);
	}
	fseek(d->handle,d->iblocks[num]->position,SEEK_SET);
	convertblock(d->iblocks[num],&scratchblock,TODISK);
	if ( mwrite(d->handle,(char *)(&scratchblock),sizeof(INDEXBLOCK)) ) {
		printf("Error while writing an index block in file %s\n",d->name);
		printf("File may be unreliable now\n");
		return(-1);
	}
	return(0);
}

/*
  	#] WriteIndexBlock :
  	#[ WriteIndex :

	Problem here is to get the links right.
*/

int WriteIndex(DBASE *d)
{
	LONG i, position;
	if ( d->iblocks == 0 ) return(0);
	for ( i = 0; i < d->info.numberofindexblocks; i++ ) {
		if ( d->iblocks[i] == 0 ) {
			printf("Error: unassigned index blocks. Cannot write\n");
			return(-1);
		}
	}
	d->info.lastindexblock = -1;
	for ( i = 0; i < d->info.numberofindexblocks; i++ ) {
		position = d->iblocks[i]->position;
		if ( position <= 0 ) {
			fseek(d->handle,0,SEEK_END);
			position = ftell(d->handle);
			d->iblocks[i]->position = position;
			if ( i <= 0 ) d->iblocks[i]->previousblock = -1;
			else d->iblocks[i]->previousblock = d->iblocks[i-1]->position;
		}
		else fseek(d->handle,position,SEEK_SET);
		convertblock(d->iblocks[i],&scratchblock,TODISK);
		if ( mwrite(d->handle,(char *)(&scratchblock),sizeof(INDEXBLOCK)) ) {
			printf("Error while writing index of file %s",d->name);
			d->iblocks[i]->position = -1;
			return(-1);
		}
		d->info.lastindexblock = position;
	}
	return(0);
}

/*
  	#] WriteIndex :
  	#[ WriteIniInfo :
*/

int WriteIniInfo(DBASE *d)
{
	INIINFO inf;
	fseek(d->handle,0,SEEK_SET);
	convertiniinfo(&(d->info),&inf,TODISK);
	if ( mwrite(d->handle,(char *)(&inf),sizeof(INIINFO)) ) {
		printf("Error while writing masterindex of file %s",d->name);
		return(-1);
	}
	return(0);
}

/*
  	#] WriteIniInfo :
  	#[ ReadIniInfo :
*/

int ReadIniInfo(DBASE *d)
{
	INIINFO inf;
	fseek(d->handle,0,SEEK_SET);
	if ( mread(d->handle,(char *)(&inf),sizeof(INIINFO)) ) {
		printf("Error while reading masterindex of file %s",d->name);
		return(-1);
	}
	convertiniinfo(&inf,&(d->info),FROMDISK);
	if ( d->info.entriesinindex < 0
	|| d->info.numberofindexblocks < 0
	|| d->info.lastindexblock < 0 ) {
		printf("The file %s is not a proper database\n",d->name);
		return(-1);
	}
	return(0);
}

/*
  	#] ReadIniInfo :
  	#[ AddToIndex :
*/

int AddToIndex(DBASE *d,LONG number)
{
	LONG i, oldnumofindexblocks = d->info.numberofindexblocks;
	LONG j, newnumofindexblocks;
	INDEXBLOCK **ib;
	time_t t = time(0);
	if ( number == 0 ) return(0);
	else if ( number < 0 ) {
		if ( d->info.entriesinindex < -number ) {
			printf("There are only %ld entries in the index of file %s\n",
			d->info.entriesinindex,d->name);
			return(-1);
		}
		d->info.entriesinindex += number;
dowrite:
		if ( WriteIniInfo(d) ) {
			d->info.entriesinindex -= number;
			printf("File may be corrupted\n");
			return(-1);
		}
	}
	else if ( d->info.entriesinindex+number <=
	NUMOBJECTS*d->info.numberofindexblocks ) {
		d->info.entriesinindex += number;
		goto dowrite;
	}
	else {
		d->info.entriesinindex += number;
		newnumofindexblocks = d->info.numberofindexblocks + ((number -
            (NUMOBJECTS*d->info.numberofindexblocks - d->info.entriesinindex))
		    +NUMOBJECTS-1)/NUMOBJECTS;
		if ( ( ib = mmalloc(sizeof(INDEXBLOCK *)*newnumofindexblocks,
		"index") ) == 0 ) return(-1);
		for ( i = 0; i < d->info.numberofindexblocks; i++ ) {
			ib[i] = d->iblocks[i];
		}		
		for ( i = d->info.numberofindexblocks; i < newnumofindexblocks; i++ ) {
			if ( ( ib[i] = mmalloc(sizeof(INDEXBLOCK),"index block") ) == 0 ) {
				for ( --i; i >= d->info.numberofindexblocks; i-- ) free(ib[i]);
				free(ib);
				return(-1);
			}
			if ( i > 0 ) ib[i]->previousblock = ib[i-1]->position;
			else ib[i]->previousblock = -1;
			for ( j = 0; j < NUMOBJECTS; j++ ) {
				ib[i]->objects[j].date = t;
				ib[i]->objects[j].size = 0;
				ib[i]->objects[j].position = -1;
			}
			fseek(d->handle,0,SEEK_END);
			ib[i]->position = ftell(d->handle);
			convertblock(ib[i],&scratchblock,TODISK);
			if ( mwrite(d->handle,(char *)(&scratchblock),sizeof(INDEXBLOCK)) ) {
				printf("Error while writing new index of file %s",d->name);
				for ( ; i >= d->info.numberofindexblocks; i-- ) free(ib[i]);
				free(ib);
				return(-1);
			}
		}
		d->info.lastindexblock = ib[newnumofindexblocks-1]->position;
		d->info.firstindexblock = ib[0]->position;
		d->info.numberofindexblocks = newnumofindexblocks;
		if ( WriteIniInfo(d) ) {
			d->info.numberofindexblocks = oldnumofindexblocks;
			d->info.entriesinindex -= number;
			printf("File may be corrupted\n");
			for ( i = d->info.numberofindexblocks; i < newnumofindexblocks; i++ )
				free(ib[i]);
			free(ib);
			return(-1);
		}
		free(d->iblocks);
		d->iblocks = ib;
	}
	return(0);
}

/*
  	#] AddToIndex :
  	#[ GetDbase :
*/

DBASE *GetDbase(char *filename)
{
	FILE *f;
	DBASE *d;
	if ( ( f = fopen(filename,"r+b") ) == 0 )
		return(NewDbase(filename,0));
/*	setbuf(f,0); */
	if ( ( d = (DBASE *)mmalloc(sizeof(DBASE),"database") ) == 0 ) {
		fclose(f);
		return(0);
	}
	d->name = str_dup(filename); /* For the moment just for the error messages */
	d->handle = f;
	if ( ReadIniInfo(d) || ReadIndex(d) ) { free(d); fclose(f); return(0); }
	d->name = str_dup(filename);
	return(d);
}

/*
  	#] GetDbase :
  	#[ OpenDbase :
*/

DBASE *OpenDbase(char *filename)
{
	FILE *f;
	DBASE *d;
	if ( ( f = fopen(filename,"r+b") ) == 0 ) {
		printf("Cannot open file %s\n",filename);
		return(0);
	}
/*	setbuf(f,0); */
	if ( ( d = (DBASE *)mmalloc(sizeof(DBASE),"database") ) == 0 ) {
		fclose(f);
		return(0);
	}
	d->name = filename; /* For the moment just for the error messages */
	d->handle = f;
	if ( ReadIniInfo(d) || ReadIndex(d) ) { free(d); fclose(f); return(0); }
	d->name = str_dup(filename);
	return(d);
}

/*
  	#] OpenDbase :
  	#[ ReadObject :
*/

char **ReadObject(DBASE *d,LONG number)
{
	OBJECTS *obj;
	LONG i, j;
	char *buffer, *s, **argv;
	if ( number >= d->info.entriesinindex ) {
		printf("Reference to non-existing object number %ld\n",number+1);
		printf("There are %ld objects\n",d->info.entriesinindex);
		return(0);
	}
	i = number/NUMOBJECTS;
	j = number%NUMOBJECTS;
	obj = &(d->iblocks[i]->objects[j]);
	if ( obj->position > 0 ) {
		fseek(d->handle,obj->position,SEEK_SET);
		if ( ( buffer = (char *)mmalloc(obj->size,"data object") ) == 0 ) return(0);
		if ( mread(d->handle,buffer,obj->size) ) {
			printf("Could not read object %ld\n",(LONG)(number+1));
			printf("Its size should have been %ld\n",obj->size);
			free(buffer); return(0);
		}
	}
	else {
		if ( ( buffer = (char *)mmalloc(4,"data object") ) == 0 ) return(0);
		*buffer = 0; buffer[1] = 0;
	}
/*
	Now we parse for the substrings: "keyword=........\0"
	We count them first.
*/
	i = 1; j = obj->size;
	s = buffer;
	while ( --j >= 0 ) {
		if ( *s ) {
			i++; s++;
			while ( --j >= 0 && *s ) s++;
		}
		s++;
	}
	if ( ( argv = mmalloc(sizeof(char *)*(i+1),"data object substrings") )
		== 0 ) {
		free(buffer); return(0);
	}
	argv[0] = buffer;
	i = 1; j = obj->size;
	s = buffer;
	while ( --j >= 0 ) {
		if ( *s ) {
			argv[i++] = s++;
			while ( --j >= 0 && *s ) { s++; }
		}
		s++;
	}
	argv[i] = 0;
	return(argv);
}

/*
  	#] ReadObject :
  	#[ GetSubString :
*/

char *GetSubString(char *keyword,char **argv)
{
	char *s1, *s2;
	argv++;
	while ( *argv ) {
		s1 = keyword; s2 = *argv;
		while ( *s1 && *s1 == *s2 ) { s1++; s2++; }
		if ( *s1 == 0 && *s2 == '=' ) return(s2+1);
		argv++;
	}
	return(0);
}

/*
  	#] GetSubString :
  	#[ WriteObject :
*/

int WriteObject(DBASE *d,char **argv,LONG number)
{
	char **a = argv, *s;
	LONG i, j, position, size;
	OBJECTS *obj;
	if ( d->mode == INPUTONLY ) {
		printf("Not allowed to write to input\n");
		return(-1);
	}
	if ( number >= d->info.entriesinindex ) {
		printf("Reference to non-existing object number %ld\n",number+1);
		return(0);
	}
	j = number/NUMOBJECTS;
	i = number%NUMOBJECTS;
	obj = &(d->iblocks[j]->objects[i]);

	fseek(d->handle,0,SEEK_END);
	position = ftell(d->handle);
	size = 0;
	while ( *(++a) ) {
		s = *a;
		i = 1; while ( *s ) { s++; i++; }
		if ( mwrite(d->handle,*a,i) ) {
			printf("Error while writing object\n");
			return(-1);
		}
		size += i;
	}
	obj->position = position;
	obj->size = size;
	obj->date = time(0);
	return(WriteIndexBlock(d,j));
}

/*
  	#] WriteObject :
  	#[ AddObject :
*/

LONG AddObject(DBASE *d,char **argv)
{
	LONG number;
	number = d->info.entriesinindex;
	if ( AddToIndex(d,1) ) return(-1);
	if ( WriteObject(d,argv,number) ) return(-1);
	return(number);
}

/*
  	#] AddObject :
  	#[ Cleanup :

	Compactify a database by copying it without its holes.
*/

char filetemplate[] = "minosXXXXXX";

int Cleanup(DBASE *d)
{
	DBASE *dd;
	LONG numblocks, i, j, maxsize;
	int error = 0, fd;
	OBJECTS *obj;
	char *buffer;
	numblocks = (d->info.entriesinindex+NUMOBJECTS-1)/NUMOBJECTS;
	if ( ( ( dd = (DBASE *)mmalloc(sizeof(DBASE),"cleanup data base") ) == 0 )
	|| ( ( dd->iblocks = (INDEXBLOCK **)
	mmalloc(numblocks*sizeof(INDEXBLOCK *),"cleanup blocks") ) == 0 ) ) {
		if ( dd ) free(dd);
		return(-1);
	}
	for ( i = 0; i < numblocks; i++ ) dd->iblocks[i] = d->iblocks[i];
	dd->info.entriesinindex = d->info.entriesinindex;
	dd->info.numberofindexblocks = numblocks;;

	dd->name = strdup(filetemplate);
	
	if ( ( fd = mkstemp(dd->name) ) == -1 ) {
		printf("Cannot create tmp file for cleanup with template %s\n",dd->name);
		free(dd->iblocks); free(dd->name); free(dd);
		return(-1);
	}
	if ( ( dd->handle = fdopen(fd,"wb+") ) == 0 ) {
/*
	dd->name = tmpnam(0);
	if ( ( dd->handle = fopen(dd->name,"r+b") ) == 0 ) {
*/
		printf("Cannot open tmp file with name %s\n",dd->name);
		free(dd->iblocks); free(dd->name); free(dd);
		return(-1);
	}
	if ( WriteIniInfo(dd) == 0 ) {
		for ( i = 0; i < numblocks; i++ ) dd->iblocks[i]->position = -1;
/*
		Reserve space for the index
*/
		if ( WriteIndex(dd) ) { /* reparation involves rereading the index */
repair:
			ReadIndex(d);
			fclose(dd->handle); remove(dd->name);
			free(dd->iblocks); free(dd->name); free(dd);
			return(-1);
		}
		dd->info.firstindexblock = dd->iblocks[0]->position;
		dd->info.lastindexblock = dd->iblocks[numblocks-1]->position;
/*
		Now copy the objects one by one
*/
		maxsize = 0;
		for ( i = 0; i < numblocks; i++ ) {
		for ( j = 0; j < NUMOBJECTS; j++ ) {
			obj = &(dd->iblocks[i]->objects[j]);
			if ( obj->size > maxsize ) maxsize = obj->size;
		} }
		if ( ( buffer = (char *)mmalloc(maxsize,"cleanup buffer") ) == 0 )
			goto repair;
		for ( i = 0; i < numblocks; i++ ) {
		for ( j = 0; j < NUMOBJECTS; j++ ) {
			obj = &(dd->iblocks[i]->objects[j]);
			fseek(d->handle,obj->position,SEEK_SET);
			if ( mread(d->handle,buffer,obj->size) ) {
				printf("Read error while copying\n");
				goto repair;
			}
			obj->position = ftell(dd->handle);
			if ( mwrite(dd->handle,buffer,obj->size) ) {
				printf("Write error while copying\n");
				goto repair;
			}
		} }
		if ( WriteIndex(dd) || WriteIniInfo(dd) ) goto repair;
/*
		Now we were successful
*/
		fclose(d->handle);
		remove(d->name);
		rename(dd->name,d->name);
		d->handle = dd->handle;
		d->info = dd->info;
		free(d->iblocks);
		d->iblocks = dd->iblocks;
		free(dd->name);
		free(dd);
		return(error);
	}
	else error = -1;
	free(dd->iblocks); free(dd->name); free(dd);
	return(error);
}

/*
  	#] Cleanup :
  	#[ NewDbase :

	Creates a new database with 'number' entries in the index.
*/

DBASE *NewDbase(char *name,LONG number)
{
	FILE *f;
	DBASE *d;
	LONG numblocks, i;
	if ( number < 0 ) number = 0;
	if ( ( f = fopen(name,"r+b") ) != 0 ) {
		printf("There is already a file with name %s\n",name);
		fclose(f);
		return(0);
	}
	numblocks = (number+NUMOBJECTS-1)/NUMOBJECTS;
	if ( ( ( d = (DBASE *)mmalloc(sizeof(DBASE),"new database") ) == 0 )
	|| ( ( d->iblocks = (INDEXBLOCK **)mmalloc(numblocks*sizeof(INDEXBLOCK *),
	"new database") ) == 0 ) ) {
		if ( d ) free(d);
		return(0);
	}
	if ( ( f = fopen(name,"w+b") ) == 0 ) {
		printf("Could not create new file %s\n",name);
		free(d);
		return(0);
	}
/*	setbuf(f,0); */
	d->name = str_dup(name);
	d->handle = f;
	d->info.entriesinindex = number;
	d->info.numberofindexblocks = numblocks;
	if ( WriteIniInfo(d) ) {
getout:
		fclose(f);
		remove(name);
		free(d);
		return(0);
	}
	for ( i = 0; i < numblocks; i++ ) {
		if ( ( d->iblocks[i] = (INDEXBLOCK *)mmalloc(sizeof(INDEXBLOCK),
		"index blocks of new database") ) == 0 ) {
			while ( --i >= 0 ) free(d->iblocks[i]);
			goto getout;
		}
		if ( i > 0 ) d->iblocks[i]->previousblock = d->iblocks[i-1]->position;
		else d->iblocks[i]->previousblock = -1;
		d->iblocks[i]->position = ftell(f);
		if ( mwrite(f,(char *)(d->iblocks[i]),sizeof(INDEXBLOCK)) ) {
			printf("Error while writing new index blocks\n");
			goto getout;
		}
	}
	if ( numblocks > 0 ) {
		d->info.firstindexblock = d->iblocks[0]->position;
		d->info.lastindexblock = d->iblocks[numblocks-1]->position;
		if ( WriteIniInfo(d) ) {
			for ( i = 0; i < numblocks; i++ ) free(d->iblocks[i]);
			goto getout;
		}
	}
	return(d);
}

/*
  	#] NewDbase :
  	#[ TouchKey :
*/

int TouchKey(DBASE *d,char *key,char *value)
{
	LONG i;
	char **argv, *val, *v;
	int error = 0;
	for ( i = 0; i < d->info.entriesinindex; i++ ) {
		if ( ( argv = ReadObject(d,i) ) == 0 ) continue;
		if ( ( val = GetSubString(key,argv) ) == 0 ) continue;
		v = value;
		while ( *v && *val == *v ) { v++; val++; }
		if ( *val == *v ) {
			d->iblocks[i/NUMOBJECTS]->objects[i%NUMOBJECTS].date = time(0);
			d->iblocks[i/NUMOBJECTS]->flags |= DIRTYFLAG;
		}
		free(argv[0]);
		free(argv);
	}
	for ( i = 0; i < d->info.numberofindexblocks; i++ ) {
		if ( ( d->iblocks[i]->flags & DIRTYFLAG ) != 0 ) {
			if ( WriteIndexBlock(d,i) == 0 )
				d->iblocks[i]->flags &= CLEANFLAG;
			else error = -1;	/* Don't know how to repair */
		}
	}
	return(error);
}

/*
  	#] TouchKey :
  	#[ DumpContents :

	Dumps the contents of the database to file or to stdout.
*/

int DumpContents(DBASE *d,char *filename,LONG from,LONG to)
{
	FILE *f;
	char **argv, **a, *s, *t, c1;
	int num;
	LONG i;
	if ( filename ) {
		if ( ( f = fopen(filename,"w") ) == 0 ) {
			printf("Cannot create file %s\n",filename);
			return(-1);
		}
	}
	else f = stdout;
	for ( i = from; i < to; i++ ) {
		fprintf(f,"%ld:\n",i+1);
		if ( ( argv = ReadObject(d,i) ) == 0 ) continue;
		if ( *argv ) {
			a = argv; a++;
			while ( *a ) {
				s = *a;
				while ( *s ) {
					num = 0;
					t = s;
					while ( ( *s != 0 ) && ( num < 75 ) ) { s++; num++; }
					if ( *s ) {
						c1 = *s; *s = 0;
						fprintf(f," %s\\\n",t);
						*s = c1;
					}
					else fprintf(f," %s\n",t);
				}
				a++;
			}
			free(argv[0]);
		}
		free(argv);
	}
	return(0);
}

/*
  	#] DumpContents :
*/
