package nl.nikhef.slcshttps.trust;

import java.security.cert.X509Certificate;
import java.util.Date;
import java.util.Hashtable;
import java.util.Vector;
import java.util.Enumeration;
import java.security.cert.CertificateException;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.CertPathValidatorException;

/**
 * This class contains all fields and methods to check the validity of a
 * certificate chain for use as a trusted server certificate, including storing
 * for reuse.
 * @author Mischa Sall&eacute;
 * @version 0.1
 * @see TrustManagerImpl
 */
public class TrustCert	{
    /** 
     * Nested class, contains the status fields relevant for an alias: whether
     * the alias was added for a (in)valid name and/or (in)valid date; the
     * certchain is not relevant for a certain alias, since it's the same for
     * all aliases. 
     * It's <CODE>protected</CODE> and not <CODE>private</CODE> so
     * that e.g. {@link TrustManagerImpl} can also access it. A status
     * field can be used to reflect the state at the moment an invalid
     * certificate was accepted but also to contain the current status of the
     * certificate.
     * @author Mischa Sall&eacute;
     * @version 0.1
     */
    protected class Status	{
	/** <CODE>true</CODE> when the hostname is valid for the certificate. */
	public boolean nameValid;
	/** <CODE>true</CODE> when one of the certificates in the chain is
	 * expired. */
	public boolean expired;
	/** <CODE>true</CODE> when one of the certificates in the chain is
	 * not yet valid. */
	public boolean notYet;

	/**
	 * Method to clone the <CODE>Status</CODE>.
	 * @return clone of the <CODE>Status</CODE>.
	 * @see #copy(TrustCert.Status)
	 */
	public Status copy()	{
	    return copy(null);
	}

	/**
	 * Method to copy the <CODE>Status</CODE>.
	 * When <CODE>outputStatus</CODE> is non-<CODE>null</CODE> it is copied
	 * into that and returned, otherwise a new <CODE>Status</CODE> is
	 * created and returned.
	 * @param outputStatus <CODE>Status</CODE> instance to put the copy in,
	 * can be <CODE>null</CODE> in which case it's ignored.
	 * @return when <CODE>outputStatus</CODE> is non-<CODE>null</CODE> it is
	 * returned containing a copy of <CODE>this</CODE>, otherwise a new
	 * <CODE>Status</CODE> containing a copy of <CODE>this</CODE>.
	 * @see #copy()
	 */
	public Status copy(Status outputStatus)	{
	    // Copy from here to outputStatus or newStatus
	    Status newStatus=(outputStatus==null ? new Status() : outputStatus);
	    newStatus.nameValid=nameValid;
	    newStatus.expired=expired;
	    newStatus.notYet=notYet;
	    return newStatus;
	}
    }

    /** Contains a {@link HostnameChecker} used for checking the hostnames. We
     * can use the same for all checking, hence it's initialized at class
     * initialization, which improves performance. */
    static HostnameChecker hostnameChecker=
	HostnameChecker.getInstance(HostnameChecker.TYPE_TLS);

    /** Holds the actual <CODE>X509Certificate</CODE>, either set using
     * constructor {@link #TrustCert(X509Certificate[])} or using
     * {@link #setCertChain(X509Certificate[])}. Note that it is equal to the
     * 0<SUP>th</SUP> element of {@link #x509Chain}. */
    private X509Certificate x509Cert;

    /** The certificate chain for the <CODE>X509Certificate</CODE>, either set
     * using constructor {@link #TrustCert(X509Certificate[])} or using
     * {@link #setCertChain(X509Certificate[])}. Note that
     * <CODE>x509Chain[0]</CODE> is equal to {@link #x509Cert}. */
    private X509Certificate[] x509Chain;

    /** Contains a list of already seen and accepted aliases
     * (hostname:portnumber) for this certificate chain, with their status at
     * the time they were accepted. */
    protected Hashtable<String, Status> knownAliases;

    /** Current status of the certificate(chain). */
    protected Status status;

    // Following is a list of certificate chain status fields: they are the same
    // for all aliases
    /** Index of the certificate with the latest
     * {@link X509Certificate#getNotBefore() notBefore} or -1 for unset. */
    protected int chainFirstIdx;
    /** <CODE>Date</CODE> represention of the latest
     * {@link X509Certificate#getNotBefore() notBefore}. */
    protected Date chainFirstDate;
    /** msec represention of the latest
     * {@link X509Certificate#getNotBefore() notBefore}. */
    protected long chainFirstMSec;
    /** Index of the certificate with the earliest
     * {@link X509Certificate#getNotAfter() notAfter} or -1 for unset. */
    protected int chainLastIdx;
    /** <CODE>Date</CODE> represention of the earliest
     * {@link X509Certificate#getNotAfter() notAfter}. */
    protected Date chainLastDate;
    /** msec represention of the earliest
     * {@link X509Certificate#getNotAfter() notAfter}. */
    protected long chainLastMSec;
    /** Index of certificate causing the chain validation to fail. Note that
     * when we catch a {@link CertPathValidatorException} which doesn't point
     * to a specific certificate, it sets it to -1, we hence use -2 for unset.*/
    protected int chainErrorIdx;
    /** <CODE>String</CODE> describing the error causing the chain validation to
     * fail. */
    protected String chainError;

    /**************************************************************************
     * METHODS
     *************************************************************************/

    /**
     * Constructs a default <CODE>TrustCert</CODE>. Its certificate chain can be
     * set using {@link #setCertChain(X509Certificate[])}. 
     * @see #TrustCert(X509Certificate[]).
     */
    private TrustCert()	{
	x509Cert=null;
	knownAliases=new Hashtable<String, Status>();
	status=new Status();
	chainFirstIdx=-1;   // cert index -1: unset
	chainLastIdx=-1;    // cert index -1: unset
	chainErrorIdx=-2;   // unchecked, note that CertPathValidatorException sets it to -1 if it has none
	chainError=null;    // no error reported
    }

    /** Constructs a <CODE>TrustCert</CODE> from a <CODE>X509Certificate</CODE>
     * chain.
     * @param certChain <CODE>X509Certificate[]</CODE> chain for this
     * <CODE>TrustCert</CODE> */
    public TrustCert(X509Certificate[] certChain)
    {
	this();
	setCertChain(certChain);
    }

    /**
     * Sets the certificate chain for this <CODE>TrustCert</CODE>.
     * @param certChain <CODE>X509Certificate[]</CODE> chain for this
     * <CODE>TrustCert</CODE>
     * @see #TrustCert(X509Certificate[]).
     */
    private void setCertChain(X509Certificate[] certChain)	{
	x509Chain=certChain;
	x509Cert=certChain[0];
    }

    /**
     * Equals method, comparing two <CODE>TrustCert</CODE>s, which are
     * considered equal when their server certificate (first in chain) is equal.
     * @param other Other <CODE>TrustCert</CODE> to compare against.
     * @return boolean indicating whether the first {@link X509Certificate} in
     * each chain is equal.
     * @see #equals(X509Certificate)
     */
    public boolean equals(TrustCert other)  {
	return this.x509Cert.equals(other.x509Cert);
    }

    /**
     * Equals method, comparing the server certificate (first in chain) against
     * a {@link X509Certificate}.
     * @param other <CODE>X509Certificate</CODE> to compare the server
     * certificate (1st in chain) against.
     * @return boolean indicating whether the server certificate is equal to
     * given <CODE>X509Certificate</CODE>.
     * @see #equals(TrustCert)
     */
    public boolean equals(X509Certificate other)  {
	return this.x509Cert.equals(other);
    }

    /**
     * Method to check whether the certificate chain is valid, that is when the
     * current time &ge; the latest not-before in the chain and &le;
     * the earliest not-after in the chain. The result is stored in
     * the {@link #status} field for later usage.
     * @return boolean indicating whether it is currently valid.
     * @throws CertificateException if the certchain is not (yet) known.
     * @see #getStatus(String)
     */
    public boolean checkValidity() throws CertificateException	{
	// Check if we know the boundaries...
	if (chainFirstIdx==-1 || chainLastIdx==-1)
	    setExtremes();

	// Get current time
	long now=new Date().getTime();

	// Do both tests separately: we could have them both wrong...
	status.notYet=status.expired=false;
	if (now < chainFirstMSec)
	    status.notYet=true;
	if (now > chainLastMSec)
	    status.expired=true;
	return !(status.expired || status.notYet);
    }

    /**
     * Method to check whether the server certificate is valid for given
     * <CODE>hostName</CODE>. The result is stored in the {@link #status} field
     * for later usage.
     * @param hostName hostname to check the server certificate against.
     * @return boolean indicating whether it is valid for given hostname.
     * @throws CertificateException in case of error (not when certificate is
     * invalid for hostname).
     * @see HostnameChecker
     * @see #getStatus(String)
     */
    public boolean checkHostname(String hostName) throws CertificateException {
	try {
	    hostnameChecker.match(hostName,x509Cert);
	    status.nameValid=true;
	} catch (CertificateException e)    {
	    status.nameValid=false;
	} catch (Exception e)	{
	    status.nameValid=false;
	    throw new CertificateException("Error processing hostname check: "+
					    e.getMessage());
	}
	return status.nameValid;
    }

    /**
     * Method to check whether the certificate chain is valid, that is in
     * particular if the whole chain can be followed back to a trusted root
     * certificate. It uses the earliest not-after date of the chain, to prevent
     * errors from time-invalidity for which we have {@link #checkValidity()}.
     * Note that if for some reason there is no time at which the entire
     * certificate chain was, is or will be valid, we still will get an error
     * here. The {@link #chainError} field will contain a (descriptive) error
     * message or be null if the chain validated.
     * @return boolean indicating whether it is valid.
     * @throws CertificateException in case of error (not when certificate is
     * invalid for hostname).
     * @see CertChainChecker
     */
    public boolean checkChain()	throws CertificateException {
	try { // Check at latest date in validity interval
	    CertChainChecker.validate(x509Chain,chainLastDate);
	    chainErrorIdx=0; // valid check
	    chainError=null; // Indicates it's all fine
	} catch (CertPathValidatorException e)	{
	    chainErrorIdx=e.getIndex();
	    chainError=e.getMessage();
	    if (chainError==null) chainError=""; // Make sure it's != null
	} catch (Exception e)	{
	    chainErrorIdx=-2; // No valid check has been done
	    chainError="Error processing certificate chain: "+e.getMessage();
	    throw new CertificateException(
		"Error processing certificate chain: "+e.getMessage());
	}
	return (chainError==null);
    }

    /**
     * Method to get the status of a known alias.
     * @param alias String representing the alias for which to get the
     * <CODE>Status</CODE> (alias is hostname:portnumber).
     * @return Status as it was when the certificate/alias was last seen.
     */
    public Status getStatus(String alias)   {
	return (Status)knownAliases.get(alias);
    }

    /**
     * Adds the given alias to the list of aliases for this
     * <CODE>TrustCert</CODE>.
     * @param alias String representing the alias to add (alias is
     * hostname:portnumber).
     * @see #removeAlias(String)
     */
    public void addAlias(String alias)   {
	knownAliases.put(alias,status.copy());
    }

    /**
     * Removes the given alias from the list of aliases for this
     * <CODE>TrustCert</CODE>.
     * @param alias String representing the alias to remove (alias is
     * hostname:portnumber).
     * @see #removeAlias(String)
     */
    public void removeAlias(String alias)   {
	knownAliases.remove(alias);
    }

    /**
     * Finds the latest not-before time and the earliest not-after time for the
     * certificate chain, thus finding the smallest interval for which the whole
     * chain is valid. It sets the outcome for the former in the fields {@link
     * #chainFirstIdx}, {@link #chainFirstDate} and {@link #chainFirstMSec} and
     * for the latter in the fields {@link #chainLastIdx}, {@link
     * #chainLastDate} and {@link #chainLastMSec}. Both the date and msec
     * representation are stored for performance.
     * @throws CertificateException when the server certificate or chain is not
     * (yet) set.
     * @see #checkValidity()
     */
    protected void setExtremes() throws CertificateException   {
	long msec;
	Date date;
	if (x509Cert==null || x509Chain==null)
	    throw new CertificateException("Certificate or chain is not yet set");
	chainFirstIdx=0;
	chainFirstDate=x509Chain[0].getNotBefore();
	chainFirstMSec=chainFirstDate.getTime();
	chainLastIdx=0;
	chainLastDate=x509Chain[0].getNotAfter();
	chainLastMSec=chainLastDate.getTime();
	for (int i=1; i<x509Chain.length; i++)	{
	    date=x509Chain[i].getNotBefore();
	    msec=date.getTime();
	    if (msec>chainFirstMSec) {
		chainFirstIdx=i;
		chainFirstMSec=msec;
		chainFirstDate=date;
	    }
	    date=x509Chain[i].getNotAfter();
	    msec=date.getTime();
	    if (msec<chainLastMSec)   {
		chainLastIdx=i;
		chainLastMSec=msec;
		chainLastDate=date;
	    }
	}
    }

    /**
     * Creates a <CODE>String</CODE> describing all the errors for given
     * <CODE>alias</CODE>, using its stored <CODE>Status</CODE>.
     * @param alias String containing the alias (hostname:portnumber).
     * @return String describing all the errors with the certificate chain for
     * given alias.
     */
    protected String getAliasErrors(String alias)	{
	// String constants to be used in the error
	final String HOSTNAME="hostname doesn't match";
	final String EXPIRED="expired chain";
	final String NOTYET="not-yet-valid chain";
	final String CHAIN="cert chain failed";

	Status status=getStatus(alias);
	StringBuffer error=new StringBuffer();
	boolean entry=false;
	if (!status.nameValid)	{
	    error.append(HOSTNAME);
	    entry=true;
	}
	if (status.expired) {
	    error.append(entry ? ", "+EXPIRED : EXPIRED);
	    entry=true;
	}
	if (status.notYet) {
	    error.append(entry ? ", "+NOTYET : NOTYET);
	    entry=true;
	}
	if (chainError!=null)	{
	    error.append(entry ? ", "+CHAIN : CHAIN);
	    entry=true;
	}
	if (entry)
	    return alias+": "+error.toString();
	else // Old one is valid
	    return null;
    }
   
    /**
     * Creates a <CODE>String</CODE> array, one for each known alias, each
     * containing a list of all the errors for that alias.
     * @return String[] one element per alias, each with an error list.
     * @see #getAliasErrors(String)
     * @see #getErrors(String)
     */
    protected String[] getOldErrors()	{
	Vector<String> errorVector=new Vector<String>();
	Enumeration<String> aliases=knownAliases.keys();
	String error;
	for (; aliases.hasMoreElements();)	{
	    error=getAliasErrors(aliases.nextElement());
	    if (error!=null) // probably not necessary, but nicer
		errorVector.add(error);
	}
	int num=errorVector.size();
	if (num==0) return null; // Don't create empty String array!
	String[] errors=new String[num];
	errorVector.toArray(errors);
	return errors;
    }
    
    /**
     * Creates a <CODE>String</CODE> array, one for each error for the current
     * connection, using the data in the {@link #status} field.
     * @param host <CODE>String</CODE> with the hostname, only used in the
     * error message.
     * @return String[] one element per error.
     * @see #getOldErrors()
     */
    protected String[] getErrors(String host)	{
	Vector<String> errorVector=new Vector<String>();
	if (!status.nameValid)
	    errorVector.add("Host \""+host+"\" is not valid for certificate");
	if (status.expired)
	    errorVector.add("Certificate chain expired "+
		chainLastDate.toString()+" (cert no. "+chainLastIdx+")");
	if (status.notYet)
	    errorVector.add("Certificate chain is not yet valid "+
		chainFirstDate.toString()+" (cert no. "+chainFirstIdx+")");
	if (chainErrorIdx!=-2 && chainError!=null)
	    errorVector.add("Certificate chain failed validation"+
		(chainErrorIdx==-1 ? ": " : " (cert no. "+chainErrorIdx+"): ")+
		chainError);

	String[] errors=new String[errorVector.size()];
	errorVector.toArray(errors);
	return errors;
    }
}

