package nl.nikhef.slcshttps.gui;

import java.awt.event.ActionListener;
import java.awt.event.ActionEvent;
import javax.swing.JPanel;
import javax.swing.JTable;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JScrollPane;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.BorderFactory;
import java.awt.Dimension;

import java.security.cert.X509Certificate;

import java.security.KeyStoreException;

import nl.nikhef.slcshttps.CAHttps;

/**
 * Class to create a panel showing information about the serial number of the
 * client side certificate in use for the default {@link
 * javax.net.ssl.SSLSocketFactory} for {@link javax.net.ssl.HttpsURLConnection}
 * or {@link nl.nikhef.slcshttps.trust.HttxURLConnection} and and buttons to
 * change it. Note {@link CAPanel} shows either one or two
 * <CODE>SerialPanel</CODE>s, one for Https, one for Httx.
 * @author Mischa Sall&eacute;
 * @version 0.1
 * @see CAPanel
 * @see nl.nikhef.slcshttps.CAHttps#setSSLSocketFactory()
 */
public class SerialPanel extends JPanel implements ActionListener {
    /** The information for this <CODE>SerialPanel</CODE> is taken from here. */
    CAHttps ca=null;
    /** Contains the serialnumber, updated by {@link #actButton} and {@link
     * #deactButton}. */
    NonEditableDefaultTableModel serialModel;
    /** Activate the certificate for client side authentication. */
    JButton actButton;
    /** Dectivate the certificate for client side authentication. */
    JButton deactButton;
    /** whether or not the Httx is shown by {@link CAPanel}, what is actually
     * shown by this <CODE>SerialPanel</CODE> is determined by the parameter in
     * {@link #SerialPanel(CAHttps,boolean)}. */
    private boolean showHttx;
    /** whether or Https <em>AND</em> Httx status are shown by {@link CAPanel},
     * what is actually shown by this <CODE>SerialPanel</CODE> is determined by
     * the parameter in {@link #SerialPanel(CAHttps,boolean)}. */
    private boolean showBoth;
    /** whether or not Https is masked by {@link CAPanel}, what is actually
     * shown by this <CODE>SerialPanel</CODE> is determined by the parameter in
     * {@link #SerialPanel(CAHttps,boolean)}, masking means: show only Httx
     * status, but update both Https and Httx. */
    private boolean maskHttps;

    /**
     * Constructs a <CODE>SerialPanel</CODE> for either Httx or Https,
     * depending on the value of <CODE>httx</CODE>.
     * @param caHttps the <CODE>CAHttps</CODE> to get the status, certificate
     * and settings from.
     * @param httx whether this panel is for Httx (<CODE>true</CODE>) or
     * Https (<CODE>false</CODE>). 
     * @throws KeyStoreException when retrieval of certificate from
     * <CODE>caHttps</CODE> fails (not when <CODE>caHttps</CODE> is
     * <CODE>null</CODE>).
     */
    public SerialPanel(CAHttps caHttps, boolean httx) throws KeyStoreException {
	// actionPerformed must be able to change caHttps and
	// know status of x509Cert, hence these fields are non-local.
	ca=caHttps;
	showHttx=httx;
	maskHttps=ca.getMaskHttps();
	showBoth=ca.getUseHttx() && ca.getUseHttps() && !maskHttps;

	// Get certificate
	X509Certificate x509Cert=null;
	try {
	    x509Cert=ca.getCertificate();
	} catch(KeyStoreException e)  {
	    x509Cert=null;
	    throw (new KeyStoreException("SerialPanel: "+e.getMessage()));
	}

	// Setup the none-editable serialnumber table: serialModel.
	// explicitly mention number of columns, as we have no headers
	serialModel = new NonEditableDefaultTableModel(0,2);
        serialModel.addRow(new String[] {
	    "Active serialnumber", getSerial(caHttps,showHttx) });

	// Create the JTable/JScrollPane containing the serialTable
	JTable serialTable = new JTable(serialModel);
	// Should not get focuss
	serialTable.setFocusable(false);
	
	// Change relative width of left and right columns:
	// when two serial panels are shown make the columns equal, otherwise
	// make them equal to the ones of the certTable.
	if (showBoth)	{
	    serialTable.getColumnModel().getColumn(0).setPreferredWidth(500);
	    serialTable.getColumnModel().getColumn(1).setPreferredWidth(500);
	} else {
	    serialTable.getColumnModel().getColumn(0).setPreferredWidth(333);
	    serialTable.getColumnModel().getColumn(1).setPreferredWidth(667);
	}

	// Change dimensions: height equal to exact table height
	Dimension serialDim=serialTable.getPreferredScrollableViewportSize();
	serialDim.height=serialTable.getRowHeight()*(serialModel.getRowCount());
	serialTable.setPreferredScrollableViewportSize(serialDim);

	// No headerline for the table.
	serialTable.setTableHeader(null);

	// topPanel contains the table and header. We create a separate panel to
	// get the position of the label right.
	JPanel topPanel = new JPanel();
	topPanel.setLayout(new BoxLayout(topPanel,BoxLayout.Y_AXIS));
	// add header for serialTable to topPanel
	if (showHttx)
	    topPanel.add(new JLabel("HTTX Client Certificate Information"));
	else
	    topPanel.add(new JLabel("HTTPS Client Certificate Information"));
	// vertical interspace
	topPanel.add(
	    Box.createRigidArea(new Dimension(0,serialTable.getRowHeight()/2)));
	// add serialTable to topPanel
	topPanel.add(new JScrollPane(serialTable));
    
	// Setup total serialpanel, which contains topPanel and buttons.
	// layout
	setLayout(new BoxLayout(this,BoxLayout.Y_AXIS));
	// add header and table panel
	add(topPanel);
	// vertical interspace
	add(Box.createRigidArea(new Dimension(0,serialTable.getRowHeight()/2)));
	// create and add button panel
	if (httx)
	    add(getButtonPanel(x509Cert!=null,ca.getCAHttxCertNo()!=null));
	else
	    add(getButtonPanel(x509Cert!=null,ca.getCAHttpsCertNo()!=null));
	// Outer borders
	setBorder(BorderFactory.createCompoundBorder(
                BorderFactory.createEtchedBorder(),
                BorderFactory.createEmptyBorder(12, 12, 12, 12)));
    }

    /**
     * Called when one of the buttons is pressed. It sets the correct
     * button labels and enables/disables the certificate. It also updates the
     * serial number in the table. It uses the instance variables
     * {@link #ca} to change the {@link javax.net.ssl.SSLSocketFactory},
     * {@link #serialModel} to change the text and {@link #actButton}
     * and {@link #deactButton} to set their status and label.
     * @param event type of event (which button has been pressed).
     */
    public void actionPerformed(ActionEvent event)	{
	// Get the event name.
	String action=event.getActionCommand();
	// Will contain the new text in the panel.
	String serialText;
	try {
	    if (action.equals("activate")) { // Activate button pressed
		if (showHttx)
		    ca.setHttxSSLSocketFactory();
		if (!showHttx || maskHttps)
		    ca.setHttpsSSLSocketFactory();
		actButton.setText("update");
		deactButton.setEnabled(true);
	    } else {
		if (action.equals("deactivate")) { // Deactivate button pressed
		    if (showHttx)   
			ca.resetHttxSSLSocketFactory();
		    if (!showHttx || maskHttps)
			ca.resetHttpsSSLSocketFactory();
		    actButton.setText("activate");
		    deactButton.setEnabled(false);
		}
	    }
	    serialText=getSerial(ca,showHttx);
	} catch (Exception e)	{ // Cannot throw (see actionPerformed def.)
	    serialText=e.getMessage();
	}
	// Update serial number in use in its table
	serialModel.setValueAt(serialText,0,1); // upperleft is 0,0
    }

    /**
     * Creates a <CODE>JPanel</CODE> containing the activate/deactivate buttons.
     * @param cert <CODE>boolean</CODE> whether there is a certificate.
     * @param active <CODE>boolean</CODE> whether a certificate is active.
     * @return <CODE>JPanel</CODE> containing the buttons.
     */
    private JPanel getButtonPanel(boolean cert, boolean active) {
	String actText="activate";	// Text on activate button
	String updText="update";	// Text on update button
	String deactText="deactivate";	// Text on deactivate button
	String longText=deactText;	// longest text on button

	// Setup buttons with longest text, they can get smaller, not bigger:
	actButton = new JButton(longText); // Create real label later
	deactButton = new JButton(longText); // Create real label later
	
	// get and set their fixed size
	Dimension buttonSize=new Dimension(actButton.getPreferredSize());
	actButton.setPreferredSize(buttonSize);
	deactButton.setPreferredSize(buttonSize);
	
	// Add listeners
        actButton.addActionListener(this);
	actButton.setActionCommand("activate");
        deactButton.addActionListener(this);
	deactButton.setActionCommand("deactivate");

	// Initialize their label and status
	deactButton.setText(deactText); // always same text on deact button
	// if there is a certificate: we can activate or update
	if (cert)
	    actButton.setEnabled(true);
	else
	    actButton.setEnabled(false);
	// if there is a serial number: we can deactivate, act button: update
	if (active)    {
	    deactButton.setEnabled(true);
	    actButton.setText(updText);
	} else	{
	    deactButton.setEnabled(false);
	    actButton.setText(actText);
	}

	// Now setup the JPanel for them
	JPanel buttonPanel=new JPanel();
	// horizontal layout
	buttonPanel.setLayout(new BoxLayout(buttonPanel,BoxLayout.LINE_AXIS));
	buttonPanel.add(Box.createHorizontalGlue()); // stretch left
	buttonPanel.add(actButton);
	buttonPanel.add(Box.createHorizontalGlue()); // stretch middle
	buttonPanel.add(deactButton);
	buttonPanel.add(Box.createHorizontalGlue()); // stretch right

	return buttonPanel;
    }

    /**
     * Gets the serial number from the {@link nl.nikhef.slcshttps.CAHttps} for
     * either {@link nl.nikhef.slcshttps.trust.HttxURLConnection} (when
     * <CODE>httx</CODE> is <CODE>true</CODE>) or
     * {@link javax.net.ssl.HttpsURLConnection} (otherwise).
     * @param caHttps <CODE>CAHttps</CODE> to use.
     * @param httx whether to return the value for Httx (<CODE>true</CODE>) or
     * Https (<CODE>false</CODE>). 
     * @return <CODE>String</CODE> representation of certificate serial number
     * in use.
     */
    private String getSerial(CAHttps caHttps,boolean httx)	{
	if (caHttps==null)
	    return CAHttps.getSerialString(null);
	else
	    return (httx ? CAHttps.getSerialString(caHttps.getCAHttxCertNo()) :
			   CAHttps.getSerialString(caHttps.getCAHttpsCertNo()));
    }
}
