package nl.nikhef.slcshttps.gui;

import javax.swing.JDialog;
import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.JLabel;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.BorderFactory;
import java.awt.CardLayout;
import java.awt.event.ActionListener;
import java.awt.event.ActionEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;


import java.security.cert.X509Certificate;
import java.util.Date;

import java.security.KeyStoreException;
import java.io.IOException;

import nl.nikhef.slcshttps.CAHttps;
import nl.nikhef.slcshttps.SURFCAHttps;
import nl.nikhef.slcshttps.SURFCAHttps.SURFCACommunicator;

/**
 * Class providing an extension to {@link JDialog} guiding the user through the
 * online CA handshake for a {@link SURFCAHttps} using a {@link CardLayout}.
 * @author Mischa Sall&eacute;
 * @version 0.1
 */
public class SURFCAInitDialog extends JDialog implements ActionListener {
    /** this will be the new {@link CAHttps}, which is an instance of {@link
     * SURFCAHttps}, but initially will be equal to the <CODE>CAHttps</CODE>
     * that is passed to the constructor; it can be retrieved using {@link
     * #getSURFCAHttps()}. */
    CAHttps surfCA=null;
    /** {@link CAHttps} instance passed into the constructor, used when
     * initialization fails or is aborted, to revert. */
    CAHttps oldSurfCA=null;
    /** Subject related text used on card 1. */
    JLabel subject1Name=null;
    /** Subject related text used on card 7. */
    JLabel subject7Name=null;
    /** Subject related text used on card 8. */
    JLabel subject8Name=null;
    /** Button used in a 1-button panel,
     * changed using {@link #actionPerformed(ActionEvent)}. */
    JButton button1;
    /** First button used in a 2-button panel,
     * changed using {@link #actionPerformed(ActionEvent)}. */
    JButton button2a;
    /** Second button used in a 2-button panel,
     * changed using {@link #actionPerformed(ActionEvent)}. */
    JButton button2b;
    /** Button panel ({@link CardLayout}),
     * changed using {@link #actionPerformed(ActionEvent)}. */
    JPanel buttonPanel;
    /** Text panel ({@link CardLayout}),
     * changed using {@link #actionPerformed(ActionEvent)}. */
    JPanel cardPanel;

    /**
     * Constructs a <CODE>SURFCAInitDialog</CODE>, which sets up the dialog for
     * the first stage, the next stages are reached using actions via {@link
     * #actionPerformed(ActionEvent)}. It takes a possibly initialized instance
     * of {@link CAHttps}, which is used to determine whether to ask for
     * confirmation and is also stored internally and returned by {@link
     * #getSURFCAHttps()} in case no new instance has been created (e.g. cancel,
     * failure).
     * @param caHttps a possibly initialized instance of <CODE>CAHttps</CODE>.
     */
    public SURFCAInitDialog(CAHttps caHttps)	{
	// Initially surfCA is the old one, might stay or become new one.
	surfCA=oldSurfCA=caHttps;

	// Set modality and title
	setModal(true);			
	setTitle("Certificate Dialog");
	// handle windowclosing: tidy up upon closing.
	addWindowListener(new WindowAdapter() {
            public void windowClosing(WindowEvent we) {
		dispose();
            }
        });

	// Create text flippanels
	cardPanel=getCards();
	CardLayout cl = (CardLayout)(cardPanel.getLayout());
	// Create button flippanels
	buttonPanel=getButtons();
	CardLayout bl = (CardLayout)(buttonPanel.getLayout());
	bl.show(buttonPanel,"DOUBLE");

	// Main content panel for JDialog
	JPanel mainPanel=new JPanel();
	mainPanel.setLayout(new BoxLayout(mainPanel,BoxLayout.Y_AXIS));
	mainPanel.add(cardPanel);
	mainPanel.add(buttonPanel);
	mainPanel.setBorder(BorderFactory.createCompoundBorder(
                BorderFactory.createEtchedBorder(),
                BorderFactory.createEmptyBorder(12, 12, 12, 12)));
	// Put to Dialog
	setContentPane(mainPanel);
	pack();
	setLocationRelativeTo(null);

	// Here starts real dialog

	// First see if there is an old one with a non-expired certificate:
	// then need confirmation
	boolean confirm=false;
	String confirmText=null;
	if (caHttps!=null)  {
	    X509Certificate x509Cert;
	    try {
		x509Cert=caHttps.getCertificate();
		if (x509Cert!=null &&
		    x509Cert.getNotAfter().getTime()>=new Date().getTime())
		{
		    confirm=true;
		    confirmText="<html>CA object is already initialized with "+
			 "a certificate with subject:"+
			 "<P>&nbsp;<P><I>"+
			 x509Cert.getSubjectX500Principal().toString()+
			 "</I></html>";
		}
	    } catch(KeyStoreException e)  {
		confirm=true;
		confirmText="<html>CA object is already initialized"+
		     "<P>&nbsp;<P><I>Failed to get its certificate</I></html>";
	    }
	}

	// Now decide first card: either confirm or equal to "init" action below
	if (confirm)	{
	    subject1Name.setText(confirmText);
	    cl.show(cardPanel,"INITCONFIRM");
	    button2a.setActionCommand("init");
	    button2a.setText("REINIT");
	    button2b.setActionCommand("cancel");
	    button2b.setText("CANCEL");
	    getRootPane().setDefaultButton(button2a);
	} else {
	    try {
		// Actual action
		surfCA=new SURFCAHttps(new EmptyCommunicator());
		// Setup for next stage "browse"
		cl.show(cardPanel, "STARTBROWS");
		button2a.setActionCommand("browse");
		button2a.setText("START BROWSER");
		button2b.setActionCommand("cancel");
		button2b.setText("CANCEL");
		getRootPane().setDefaultButton(button2a);
	    } catch (Exception e)   {
		cl.show(cardPanel, "INITFAIL");
		button2a.setActionCommand("init");
		button2a.setText("RETRY");
		button2b.setActionCommand("cancel");
		button2b.setText("CANCEL");
		getRootPane().setDefaultButton(button2a);
	    }
	}
	setVisible(true);
    }

    /**
     * Called when <CODE>SURFCAInitDialog</CODE> receives an
     * <CODE>ActionEvent</CODE>. It changes to the correct text/button cards and
     * sets their text where needed. 
     * @param event <CODE>ActionEvent</CODE> received
     */
    public void actionPerformed(ActionEvent event)  {
	String action=event.getActionCommand();

	CardLayout cl = (CardLayout)(cardPanel.getLayout());
	CardLayout bl = (CardLayout)(buttonPanel.getLayout());
	// First stage: initialization of SURFCAHttps object
	if ("init".equals(action))  {
	    try {
		// Actual action
		surfCA=new SURFCAHttps(new EmptyCommunicator());
		// Setup for next stage "browse"
		cl.show(cardPanel, "STARTBROWS");
		button2a.setActionCommand("browse");
		button2a.setText("START BROWSER");
		button2b.setActionCommand("cancel");
		button2b.setText("CANCEL");
		getRootPane().setDefaultButton(button2a);
	    } catch (Exception e)   {
		cl.show(cardPanel, "INITFAIL");
		button2a.setActionCommand("init");
		button2a.setText("RETRY");
		button2b.setActionCommand("cancel");
		button2b.setText("CANCEL");
		getRootPane().setDefaultButton(button2a);
	    }
	// Next stage: about to start browser
	} else if ("browse".equals(action))   {
	    try {
		// Actual action
		((SURFCAHttps)surfCA).initialize();
		// Setup for next stage "cert"
		cl.show(cardPanel, "BROWSCOMPLETE");
		button2a.setActionCommand("cert");
		button2a.setText("DOWNLOAD CERTIFICATE");
		button2b.setActionCommand("cancel");
		button2b.setText("CANCEL");
		getRootPane().setDefaultButton(button2a);
	    } catch (Exception e) {
		cl.show(cardPanel, "BROWSFAIL");
		button2a.setActionCommand("browse");
		button2a.setText("RETRY");
		button2b.setActionCommand("cancel");
		button2b.setText("CANCEL");
		getRootPane().setDefaultButton(button2a);
	    }
	// Finish stage: finished with browser, download cert & set SSL
	} else if ("cert".equals(action) || "https".equals(action))  {
	    if ("cert".equals(action))	{
		try {
		    // Action
		    ((SURFCAHttps)surfCA).storeCertificate();
		} catch(Exception e)  {
		    // cert failed: retry "cert" or cancel
		    cl.show(cardPanel, "CERTFAIL");
		    button2a.setActionCommand("cert");
		    button2a.setText("RETRY");
		    button2b.setActionCommand("cancel");
		    button2b.setText("CANCEL");
		    getRootPane().setDefaultButton(button2a);
		    return;
		}
	    }
	    // When "cert" was successful, also try "https"
	    try {
		// Action
		// Uses settings from CAHttps to get the correct one(s)
		surfCA.setSSLSocketFactory();
		// Setup for next stage "close"
		cl.show(cardPanel, "CLOSE");
		bl.show(buttonPanel,"SINGLE");
		button1.setActionCommand("close");
		button1.setText("OK");
		getRootPane().setDefaultButton(button1);
	    } catch(Exception e) {
		// https failed: retry "https" or cancel
		cl.show(cardPanel, "HTTPSFAIL");
		button2a.setActionCommand("https");
		button2a.setText("RETRY");
		button2b.setActionCommand("cancel");
		button2b.setText("CANCEL");
		getRootPane().setDefaultButton(button2a);
	    }
	} else if ("cancel".equals(action))  {
	    // restore previous one
	    surfCA=oldSurfCA;
	    dispose();
	} else if ("close".equals(action))  {
	    // Close: stop all actions and quit
	    dispose();
	}
    }

    /**
     * Creates a <CODE>JPanel</CODE> with all the different textcards ({@link
     * CardLayout}). Flipping through them is done by {@link
     * #actionPerformed(ActionEvent)} which reacts on an {@link ActionEvent}
     * from a button.
     * @return JPanel containing all the different text flipcards.
     */
    private JPanel getCards()	{
	JPanel cards = new JPanel(new CardLayout());

	// Panel1: "INITCONFIRM": Confirmation of overwriting.
	JPanel panel1=new JPanel();
	panel1.setLayout(new BoxLayout(panel1,BoxLayout.Y_AXIS));
	// Set dummy text
	subject1Name=new JLabel("<html>CA object is already initialized"+
				"<P>&nbsp;"+
				"<P><I>&nbsp;</I></html>");
	panel1.add(subject1Name);
	panel1.add(new JLabel("<html><P>&nbsp;<P>Please press REINIT to "+
			      "reinitialize or CANCEL to cancel.</html>"));
	cards.add(panel1,"INITCONFIRM");

	// Panel2: "INITFAIL": Initialization failed
	JPanel panel2=new JPanel();
	panel2.setLayout(new BoxLayout(panel2,BoxLayout.Y_AXIS));
	panel2.add(new JLabel("<html>Initialization failed<P>&nbsp;"+
			      "<P>Check that following property is set and valid:"+
			      "<PRE>   "+SURFCAHttps.AUTH_URL_PROPERTY+
			      "<BR>   <I>"+SURFCAHttps.AUTH_URL+"</I>"+
			      "</html>"));
	cards.add(panel2,"INITFAIL");

	// Panel3: "STARTBROWS": About to start webbrowser
	JPanel panel3=new JPanel();
	panel3.setLayout(new BoxLayout(panel3,BoxLayout.Y_AXIS));
	panel3.add(new JLabel("<html>A new webbrowser or browsertab is about "+
			      "to start."+
			      "<P>&nbsp;<P>You will be asked to choose your "+
			      "Identity Provider and <I>log in</I>."+
			      "<P>&nbsp;"+
			      "<P><I><FONT color=\"red\">After</FONT></I> "+
			      "you are done, please go <I>back</I> to this "+
			      "window.</html>"));
	cards.add(panel3,"STARTBROWS");

	// Panel4: "BROWSFAIL": Webbrowser failed
	JPanel panel4=new JPanel();
	panel4.setLayout(new BoxLayout(panel4,BoxLayout.Y_AXIS));
	panel4.add(new JLabel("<html>Browser failed to start<P>&nbsp;"+
			      "<P>Check also that the following property is set "+
			      "and valid:"+
			      "<PRE>   "+SURFCAHttps.AUTH_URL_PROPERTY+
			      "<BR>   <I>"+SURFCAHttps.AUTH_URL+"</I></PRE>"+
			      "</html>"));
	cards.add(panel4,"BROWSFAIL");

	// Panel5: "BROWSCOMPLETE": Just-after-webbrowser
	JPanel panel5=new JPanel();
	panel5.setLayout(new BoxLayout(panel5,BoxLayout.Y_AXIS));
	panel5.add(new JLabel("<html>When <I><FONT color=\"red\">finished </FONT>"+
			      "</I>in the webbrowser,"+
			      "<P>&nbsp;<P>press DOWNLOAD CERTIFICATE."+
			      "<P>&nbsp;<P>This will download and install "+
			      "your Certificate</html>"));
	cards.add(panel5,"BROWSCOMPLETE");

	// Panel6: "CERTFAIL": Certificate download failed
	JPanel panel6=new JPanel();
	panel6.setLayout(new BoxLayout(panel6,BoxLayout.Y_AXIS));
	panel6.add(new JLabel("<html>Certificate download "+
			      "<FONT color=\"red\">failed</FONT>.<P>"+
			      "<P>Make sure you have "+
			      "<I>successfully logged in</I>"+
			      "<BR>using your webbrowser.</html>"));
	cards.add(panel6,"CERTFAIL");

	// Panel7: "HTTPSFAIL": SSLSocketFactory failed
	JPanel panel7=new JPanel();
	panel7.setLayout(new BoxLayout(panel7,BoxLayout.Y_AXIS));
	panel7.add(new JLabel("<html>Certificate downloaded succesfully<P>&nbps;"));
	subject7Name=new JLabel("");
	panel7.add(subject7Name);
	panel7.add(new JLabel("<P>&nbsp;<P>Setting HTTPS "+
			      "<FONT color=\"red\">failed</FONT>.</html>"));
	cards.add(panel7,"HTTPSFAIL");

	// Panel8: "CLOSE": Certificate installed successfully
	JPanel panel8=new JPanel();
	panel8.setLayout(new BoxLayout(panel8,BoxLayout.Y_AXIS));
	panel8.add(new JLabel("<html>Successfully installed certificate<P>&nbsp;"));
	subject8Name=new JLabel("");
	panel8.add(subject8Name);
	cards.add(panel8,"CLOSE");
	
	// Set borders
	cards.setBorder(BorderFactory.createCompoundBorder(
                BorderFactory.createEtchedBorder(),
                BorderFactory.createEmptyBorder(12, 12, 12, 12)));

	return cards;
    }

    /**
     * Creates a <CODE>JPanel</CODE> with two cards (a single and a double
     * button card) in a {@link CardLayout}, containing the different buttons
     * for the textcards. The cards and buttons are set by {@link
     * #actionPerformed(ActionEvent)}.
     * @return JPanel containing all the different button flipcards.
     */
    private JPanel getButtons()	{
	// button1: "SINGLE" panel
	// Initially: START
	button1=new JButton("START");
	button1.addActionListener(this);
	button1.setActionCommand("init");

	// button2a and button2b: "DOUBLE" panel
	// Initially: REINIT and CANCEL
	button2a=new JButton("REINIT");
	button2a.addActionListener(this);
	button2a.setActionCommand("reinit");
	button2b=new JButton("CANCEL");
	button2b.addActionListener(this);
	button2b.setActionCommand("cancel");

	// First panel: 1 button
	JPanel panel1=new JPanel();
	panel1.add(button1);

	// Second panel: 2 buttons
	JPanel panel2=new JPanel();
	panel2.setLayout(new BoxLayout(panel2,BoxLayout.LINE_AXIS));
	panel2.add(Box.createHorizontalGlue());
	panel2.add(button2a);
	panel2.add(Box.createHorizontalGlue());
	panel2.add(button2b);
	panel2.add(Box.createHorizontalGlue());

	// Add all to cards
	JPanel cards = new JPanel(new CardLayout());
	cards.add(panel1,"SINGLE"); // one button
	cards.add(panel2,"DOUBLE"); // two buttons

	return cards;
    }

    /**
     * Static method to dislay a new <CODE>SURFCAInitDialog</CODE>. 
     * @param caHttps a potentially initialized <CODE>CAHttps</CODE> instance
     * which will be returned if the dialog is broken off (cancel).
     * @return a new {@link SURFCAHttps} in case the dialog was completely
     * successfully, the given <CODE>caHttps</CODE> in case it was broken off or
     * <CODE>null</CODE> in case of error.
     */
    public static CAHttps getDialog(CAHttps caHttps)	{
	try {
	    SURFCAInitDialog dialog=new SURFCAInitDialog(caHttps);
	    return dialog.getSURFCAHttps();
	} catch (Exception e)	{
	}
	return null;
    }

    /**
     * Getter method to for the internal {@link CAHttps}.
     * @return internal {@link CAHttps}
     */
    private CAHttps getSURFCAHttps()    {
	return surfCA;
    }

    /**
     * This is a (private) almost trivial implementation of {@link
     * SURFCACommunicator} which is needed since all communication is done
     * through the {@link SURFCAInitDialog}.
     * @author Mischa Sall&eacute;
     * @version 0.1
     */
    private class EmptyCommunicator implements SURFCACommunicator {
	/**
	 * Trivial method, called just before the webbrowser is started.
	 * @throws IOException never thrown
	 */
	public void preBrowse() throws IOException  {}
	/**
	 * Trivial method, called just after the webbrowser is started.
	 * @throws IOException never thrown
	 */
	public void postBrowse() throws IOException {}
	/**
	 * Trivial method, called when an error occurs.
	 * @param text ignored
	 * @param e ignored
	 */
	public void error(String text,Exception e) {}
	/**
	 * Trivial method, called when an error occurs which might be fixed by
	 * the user, always returning false meaning do not retry.
	 * @param text ignored
	 * @param e ignored
	 * @return false
	 */
	public boolean retry(String text,Exception e) { return false; }
	/**
	 * Called when the interaction has been successful and the certificate
	 * stored, which is just before the last textcard is shown is {@link
	 * SURFCAInitDialog}, for which it sets an appropriate text about the
	 * Subject DN.
	 * @param dummy <CODE>String</CODE> - ignored.
	 */
	public void success(String dummy) {
	    X509Certificate x509Cert;
	    String text=null;
	    try {
		x509Cert=surfCA.getCertificate();
		if (x509Cert==null)
		    text="<html><I>Certificate is empty</I></html>";
		else
		    text="<html>Subject information:<P><I>"+
			 x509Cert.getSubjectX500Principal().toString()+
			 "</I></html>";
	    } catch(KeyStoreException e)  {
		text="<html><I>Failed to get certificate</I></html>";
	    }
	    subject7Name.setText(text);
	    subject8Name.setText(text);
	}
    }
}
    
