package nl.nikhef.slcshttps.gui;

import java.awt.Dimension;
import javax.swing.JPanel;
import javax.swing.BoxLayout;
import javax.swing.Box;
import java.security.KeyStoreException;

import nl.nikhef.slcshttps.CAHttps;

/**
 * Class to create a panel showing certificate information ({@link CertPanel})
 * and a panel with buttons to set or reset the default {@link
 * javax.net.ssl.SSLSocketFactory} for {@link javax.net.ssl.HttpsURLConnection}
 * and/or {@link nl.nikhef.slcshttps.trust.HttxURLConnection} ({@link
 * SerialPanel}).
 * Whether only HTTPS, only HTTX or both are used can be set using the
 * properties set for {@link CAHttps}.
 * @author Mischa Sall&eacute;
 * @version 0.1
 * @see CAHttps
 * @see CATool#showCATool(CAHttps)
 */
public class CAPanel extends JPanel {
    /**
     * Constructs a <CODE>CAPanel</CODE> with a {@link CertPanel} and one or more
     * {@link SerialPanel} for {@link javax.net.ssl.HttpsURLConnection} and/or
     * {@link nl.nikhef.slcshttps.trust.HttxURLConnection}.
     * @param caHttps the CA to get the certificate from, also used to decide
     * whether we show only Https, only Httx or both.
     * @throws KeyStoreException in case the certificate information cannot be
     * retrieved.
     */
    public CAPanel(CAHttps caHttps) throws KeyStoreException {
	// Vertical column of different panels
	setLayout(new BoxLayout(this,BoxLayout.Y_AXIS));

	// certificate panel
	try {
	    add(new CertPanel(caHttps));
	} catch (KeyStoreException e)	{
	    throw e;
	}

	// (de)activate buttons panel
	if (caHttps!=null)  {
	    try {
		// Get https/httx settings
		boolean https=caHttps.getUseHttps();
		boolean httx=caHttps.getUseHttx();
		boolean mask=caHttps.getMaskHttps();
		if (https && httx && !mask) {
		    // show both: create 'serialpanel'
		    JPanel serialPanels=new JPanel();
		    serialPanels.setLayout(
			new BoxLayout(serialPanels,BoxLayout.X_AXIS));
		    // create subpanels: 2nd arg: =?httx
		    SerialPanel httpsPanel=new SerialPanel(caHttps,false);
		    SerialPanel httxPanel=new SerialPanel(caHttps,true);
		    // update sizes, factor 3.55 is empirical ?!
		    Dimension size;
		    size=httpsPanel.getPreferredSize(); size.width/=3.55;
		    httpsPanel.setPreferredSize(size);
		    size=httxPanel.getPreferredSize(); size.width/=3.55;
		    httxPanel.setPreferredSize(size);
		    // Add subpanels to superpanel
		    serialPanels.add(httpsPanel);
		    serialPanels.add(httxPanel);
		    // Add serialPanel
		    add(serialPanels);
		} else	// show one: only when ONLY https we show https
		    add(new SerialPanel(caHttps,httx));
	    } catch (KeyStoreException e)   {
		throw e;
	    }
	}
    }
}
