package nl.nikhef.slcshttps.crypto;

import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.KeyManager;
import javax.net.ssl.TrustManager;

import java.security.KeyStoreException;
import java.security.KeyManagementException;

/**
 * The main task of this class is to provide a {@link SSLSocketFactory} for a
 * given {@link CryptoStore} containing a client certificate and corresponding
 * private key, this is the input needed for setting up Client Side
 * authentication during SSL setup.
 * @author Mischa Sall&eacute;
 * @version 0.1
 */
public class CryptoSSL {
    /** Contains the {@link KeyManager}s to be used for client side
     * authentication. */
    private KeyManager[] keyManagers=null;
    /** Contains the {@link TrustManager}s which provide authentication of
     * server side. */
    private TrustManager[] trustManagers=null;

    /**
     * constructs a new <CODE>CryptoSSL</CODE>, initializing
     * the {@link #keyManagers} field using the given {@link CryptoStore}, while
     * using the default {@link TrustManager}.
     * Initialization is done using {@link #init(CryptoStore)}, so we can
     * reinitialize the class.
     * @param cryptoStore {@link CryptoStore} to get the {@link KeyManager}[]
     * from.
     * @throws KeyStoreException
     * @see #CryptoSSL(CryptoStore,TrustManager)
     * @see #init(CryptoStore)
     */
    public CryptoSSL(CryptoStore cryptoStore) throws KeyStoreException
    {
	init(cryptoStore,null);
    }

    /**
     * constructs a new <CODE>CryptoSSL</CODE>, initializing
     * the {@link #keyManagers} field using the given {@link CryptoStore},
     * parameter and the {@link #trustManagers} field using the given
     * {@link TrustManager}. 
     * Initialization is done using #init(CryptoStore,TrustManager)}, so we can
     * reinitialize the class.
     * @param cryptoStore {@link CryptoStore} to get the {@link KeyManager}[]
     * from.
     * @param trustManager {@link TrustManager} to use for server certificate
     * checking. Use <CODE>null</CODE> for the default.
     * @throws KeyStoreException
     * @see #init(CryptoStore,TrustManager)
     */
    public CryptoSSL(CryptoStore cryptoStore,TrustManager trustManager)
	throws KeyStoreException
    {
	init(cryptoStore,trustManager);
    }

    /**
     * initializes the {@link #keyManagers} field using the give {@link
     * CryptoStore} and the {@link #trustManagers} field to use the default
     * {@link TrustManager}.
     * @param cryptoStore {@link CryptoStore} to get the {@link KeyManager}[]
     * from.
     * @throws KeyStoreException
     * @see #init(CryptoStore,TrustManager)
     */
    public void init(CryptoStore cryptoStore)
	throws KeyStoreException
    {
	init(cryptoStore,null);
    }

    /**
     * initializes the {@link #keyManagers} and {@link #trustManagers} fields
     * using the given {@link CryptoStore} and {@link TrustManager} parameters.
     * @param cryptoStore {@link CryptoStore} to get the {@link KeyManager}[]
     * from.
     * @param trustManager {@link TrustManager} to use for server
     * certificate checking. Use <CODE>null</CODE> for the default.
     * @throws KeyStoreException
     */
    public void init(CryptoStore cryptoStore,TrustManager trustManager)
	throws KeyStoreException
    {
	try {
	    keyManagers=cryptoStore.getKeyManagers();
	} catch (Exception e)	{
	    throw new KeyStoreException("Cannot get KeyManager[] from cryptoStore",e);
	}
	if (trustManager!=null)
	    trustManagers=new TrustManager[]{trustManager};
    }


    /**
     * Adds an extra {@link KeyManager} for the given {@link CryptoStore}
     * parameter. Hence multiple client certificates can be offered when setting
     * up an SSL connection.
     * @param cryptoStore {@link CryptoStore} to use to get an extra {@link
     * KeyManager} for client certificate checking from.
     * @throws KeyStoreException when getting the <CODE>KeyManagers</CODE> from
     * <CODE>cryptoStore</CODE> failed.
     */
    public void addCryptoStore(CryptoStore cryptoStore)
	throws KeyStoreException
    {
	// Save the old array
	KeyManager[] newKeyManagers,oldKeyManagers=keyManagers;
	try {
	    // Try getting an array of KeyManagers from cryptoStore
	    newKeyManagers=cryptoStore.getKeyManagers();
	} catch (Exception e)	{
	    throw new KeyStoreException(
		"Cannot get KeyManager[] from cryptoStore",e);
	}
	int length0=oldKeyManagers.length;
	int length1=newKeyManagers.length;
	// Create combined array
	keyManagers=new KeyManager[length0+length1];
	System.arraycopy(oldKeyManagers,0,keyManagers,0,length0);
	System.arraycopy(newKeyManagers,0,keyManagers,length0,length1);
    }

    /**
     * Creates a {@link SSLSocketFactory}, which provides both key and trust
     * material during the setting up of an SSL session. Key material is taken
     * from {@link #keyManagers}, trust material from {@link #trustManagers}s.
     * @return SSLSocketFactory
     * @throws KeyManagementException in case of failure
     */
    public SSLSocketFactory getSSLSocketFactory()
	throws KeyManagementException 
    {
	SSLSocketFactory socketFactory=null;
	try {
	    SSLContext sslContext=SSLContext.getInstance("TLS");
	    sslContext.init(keyManagers,trustManagers,null);
	    socketFactory=sslContext.getSocketFactory();
	} catch (Exception e)	{
	    throw new KeyManagementException(
		"Cannot get a SocketFactory for given managers",e);
	}
	return socketFactory;
    }
}
