package nl.nikhef.slcshttps;

import java.security.cert.X509Certificate;
import java.io.PrintStream;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.BufferedReader;
import java.net.HttpURLConnection; // Needed for http response code fields

import java.io.IOException;
import java.security.cert.CertificateException;
import java.security.KeyStoreException;
import java.security.KeyManagementException;

import nl.nikhef.slcshttps.crypto.CSR;
import nl.nikhef.slcshttps.util.BareBonesBrowserLaunch;
import nl.nikhef.slcshttps.util.ConsoleTools;
import nl.nikhef.slcshttps.gui.GraphTools;
import nl.nikhef.slcshttps.gui.SURFCAPopupComm;

/**
 * Implementation of the <CODE>abstract</CODE> {@link CAHttps} for obtaining a
 * certificate from the <A HREF="http://www.surfnet.nl/">SURFnet</A> online CA.
 * This typically consists of calling
 * <UL><LI>{@link #initialize()} - which start a webbrowser pointing to the
 * address set using the property {@value AUTH_URL_PROPERTY} with a hash of the
 * Certificate Signing Request ({@link CSR}).
 * <LI>{@link #storeCertificate()} - which sends the {@link CSR} as a pemstring
 * to the address set using the property {@value CERT_URL_PROPERTY}.
 * <LI>{@link #setSSLSocketFactory()} - which sets the default {@link
 * javax.net.ssl.SSLSocketFactory} to use the certificate for client side
 * authentication. This can be combined with the previous by using
 * <CODE>storeCertificate(true)</CODE>.
 * </UL>
 * @author Mischa Sall&eacute;
 * @version 0.2
 */
public class SURFCAHttps extends CAHttps {
    /** Name of property defining which {@link SURFCACommunicator} to use:
     * {@value}. Valid values of this property are
     * <UL><LI><CODE>"stdio"</CODE> use <CODE>stdio/stderr</CODE>
     * <LI><CODE>"popup"</CODE> use (swing) popups</UL>
     * @see #setCommunicator(String)
     */
    public final static String COMMPROP="nl.nikhef.slcshttps.comm";

    /** Name of property defining where to get the certificate from: {@value}; use
     * for example <CODE>"https://knikker.surfnet.nl/onlineca/x509.php"</CODE> as
     * its value. */
    public static final String CERT_URL_PROPERTY="nl.nikhef.slcshttps.CERT_URL";
    /** Name of property defining where to send the CSR hash to via webbrowser:
     * {@value}; use for example
     * <CODE>"https://knikker.surfnet.nl/onlineca/x509.php?hash="</CODE> or just
     * <CODE>"?hash="</CODE> as its value. */
    public static final String AUTH_URL_PROPERTY="nl.nikhef.slcshttps.AUTH_URL";

    /** URL to send the CSR hash to, its value is set using the property
     * {@link #AUTH_URL_PROPERTY}. */
    public static final String AUTH_URL;
    /** URL to send the CSR itself to, its value is be set using the property
     * {@link #CERT_URL_PROPERTY}. */ 
    public static final String CERT_URL;
    /** Initialize the fields {@link #AUTH_URL} and {@link #CERT_URL} using {@link
     * #AUTH_URL_PROPERTY} and {@link #CERT_URL_PROPERTY} */
    static {
	CERT_URL=System.getProperty(CERT_URL_PROPERTY);
	String url=System.getProperty(AUTH_URL_PROPERTY);
	if (url==null ||
	    url.startsWith("http://") ||
	    url.startsWith("https://"))
	    AUTH_URL=url;
	else
	    AUTH_URL=CERT_URL+url;
    }
   
    /** Contains the the value of the property {@value COMMPROP}.
     * @see #getCommunicator() */
    private static String commString=null;
    /** The default {@link SURFCACommunicator} to be used for new instances of
     * <CODE>SURFCAHttps</CODE>, can be set using {@link
     * #setCommunicator(String)}. */
    private static SURFCACommunicator defaultComm=null;
    /** Initialize <CODE>commString</CODE> and <CODE>defaultComm</CODE> */
    static {
	// Valid options: stdio, popup, make it static for the class...
	String commProp=System.getProperty(COMMPROP);
	setCommunicator(commProp);
    }
    /** Defines whether to acknowledge successful certificate import, download
     * etc. Valid options are <CODE>true</CODE> or <CODE>false</CODE>. The default
     * is same as the value in the superclass {@link CAHttps#showSuccess} which
     * in turn is set by {@link CAHttps#SUCCESS_PROP}, but it can be overridden.
     * @see #getShowSuccess()
     * @see #setShowSuccess(boolean) */
    private static boolean showSuccess=CAHttps.showSuccess;
    /** The {@link SURFCACommunicator} to be used for this instance, either
     * equal to {@link #defaultComm} or set using the constructor {@link
     * #SURFCAHttps(SURFCACommunicator)}. */
    private SURFCACommunicator comm=null;

    /**************************************************************************
     * METHODS
     *************************************************************************/

    /**
     * constructs a default <CODE>SURFCAHttps</CODE> object, which includes
     * creation of a Certificate Signing Request ({@link CSR}).
     * @throws KeyStoreException when initialization failed.
     * @see #SURFCAHttps(SURFCACommunicator)
     */
    public SURFCAHttps() throws KeyStoreException   {
	this(null,null,null,null);
    }
    
    /**
     * constructs a <CODE>SURFCAHttps</CODE> object and will use
     * <CODE>communicator</CODE> for communication with the user.
     * @param communicator {@link SURFCACommunicator} to use for this
     * SURFCAHttps instance.
     * @throws KeyStoreException when initialization failed.
     */
    public SURFCAHttps(SURFCACommunicator communicator) throws KeyStoreException
    {
	this(null,null,null,communicator);
    }
    
    /**
     * constructs a <CODE>SURFCAHttps</CODE> object and changes
     * <CODE>stdout/stdin</CODE> to the streams specified.
     * @param myErr use this stream instead of <CODE>stderr</CODE>,
     * <CODE>null</CODE> for <CODE>stderr</CODE>
     * @param myOut use this stream instead of <CODE>stdout</CODE>,
     * <CODE>null</CODE> for <CODE>stdout</CODE>.
     * @param myIn use this stream instead of <CODE>stdin</CODE>,
     * <CODE>null</CODE> for <CODE>stdin</CODE>.
     * @param communicator {@link SURFCACommunicator} to use for this
     * SURFCAHttps instance.
     * @throws KeyStoreException when initialization failed.
     */
    public SURFCAHttps(PrintStream myErr, PrintStream myOut, InputStream myIn,
		       SURFCACommunicator communicator)
	throws KeyStoreException
    {
	super(myErr,myOut,myIn);
	// Set communicator or use the default
	comm=(communicator==null ? defaultComm : communicator);
	// Do only here, if we don't use SURFCAHttps after all, there is no need
	// to set the property...
	if (AUTH_URL==null)
	    throw new RuntimeException("Property "+AUTH_URL_PROPERTY+" is not set");
	if (CERT_URL==null)
	    throw new RuntimeException("Property "+CERT_URL_PROPERTY+" is not set");
    }

    /**
     * method to initialize contact with the CA: this consists of sending the
     * SHA1 hash of the {@link CSR} via a HTTP GET to a URL, constructed using
     * the property {@value AUTH_URL_PROPERTY}, using a webbrowser, which then
     * redirects via Shibboleth to a Shibboleth IdP. The actual URL consists of
     * the value of {@value AUTH_URL_PROPERTY} plus the CSR hash.
     * @throws IOException if something has gone wrong
     * @see #initialize(String)
     * @see SURFCACommunicator
     */
    public void initialize() throws IOException
    {
	initialize(AUTH_URL);
    }

    /**
     * method to initialize contact with the CA: this consists of sending the
     * SHA1 hash of the {@link CSR} via a HTTP GET to a URL, constructed using
     * the parameter <CODE>authURL</CODE> using a webbrowser, which then
     * redirects via Shibboleth to a Shibboleth IdP. . The actual URL consists of
     * the value of {@value AUTH_URL_PROPERTY} plus the CSR hash.
     * @param authURL <CODE>String</CODE> representation of the base URL where
     * to authenticate, the CSR hash will be added to this.
     * @throws IOException if something has gone wrong
     * @see SURFCACommunicator
     * @see BareBonesBrowserLaunch
     */
    public void initialize(String authURL) throws IOException
    {
	// Pre-browser launch user interaction
	comm.preBrowse(); // Throws IOException upon error

	// Get the CSR from the cryptoStore
	CSR csr=cryptoStore.getCSR();
	if (csr==null)	{
	    comm.error("Cannot find a Certificate Signing request in "+
			       "the CryptoStore",null);
	    throw new IOException("Empty CSR in CryptoStore, cannot send");
	}

	// Create the URL to use
	String urlString="";
	try {
	    urlString=authURL+csr.hash();
	} catch (IOException e)	{
	    comm.error("Failed to get hash for Certificate Signing Request "+
		"from CryptoStore",e);
	    throw e;
	}
	// Browser launch retry loop
	while (true) {
	    try { // Launch webbrowser
		BareBonesBrowserLaunch.openURL(urlString);
		break;
	    } catch (Exception e)  {
		if (!comm.retry("Cannot open webbrowser to:\n"+
					"<I>"+urlString+"</I>",e));
		    throw new IOException(e.getMessage());
	    }
	}
	// Post-browser launch user interaction
	comm.postBrowse(); // Throws IOException upon error
    }

    /**
     * method to retrieve the certificate from the CA after successful
     * authentication. It sends the full {@link CSR}, pemencoded. It expects as
     * reply a pem encoded certificate, which will be stored in the internal
     * {@link crypto.CryptoStore}. It uses the <CODE>CERT_URL</CODE> from the
     * property {@value CERT_URL_PROPERTY} to talk to and does
     * <EM>NOT</EM> set the {@link javax.net.ssl.SSLSocketFactory}.
     * @throws IOException
     * @throws CertificateException
     * @throws KeyStoreException
     * @throws KeyManagementException
     * @see #storeCertificate(String,boolean)
     */
    public void storeCertificate()
	throws IOException, CertificateException, KeyStoreException,
	       KeyManagementException 
    {
	storeCertificate(CERT_URL,false);
    }

    /**
     * method to retrieve the certificate from the CA after successful
     * authentication. It sends the full {@link CSR}, pemencoded. It expects as
     * reply a pem encoded certificate, which will be stored in the internal
     * {@link crypto.CryptoStore}. It uses the <CODE>CERT_URL</CODE> from the
     * property {@value CERT_URL_PROPERTY} to talk to and optionally sets
     * the {@link javax.net.ssl.SSLSocketFactory}.
     * @param set <CODE>boolean</CODE> whether or not to set the
     * <CODE>SSLSocketFactory</CODE> to use the just downloaded certificate.
     * @throws IOException
     * @throws CertificateException
     * @throws KeyStoreException
     * @throws KeyManagementException when setting the
     * <CODE>SSLSocketFactory</CODE> fails 
     * @see #storeCertificate(String,boolean)
     */
    public void storeCertificate(boolean set)
	throws IOException, CertificateException, KeyStoreException,
	       KeyManagementException 
    {
	storeCertificate(CERT_URL,set);
    }

    /**
     * method to retrieve the certificate from the CA after successful
     * authentication. It sends the full {@link CSR}, pemencoded. It expects as
     * reply a pem encoded certificate, which will be stored in the internal
     * {@link crypto.CryptoStore}. It uses the parameter <CODE>certURL</CODE> to
     * talk to. It does <EM>not</EM> set the {@link javax.net.ssl.SSLSocketFactory}.
     * @param certURL URL to send the CSR to.
     * @throws IOException
     * @throws CertificateException
     * @throws KeyStoreException
     * @throws KeyManagementException
     */
    public void storeCertificate(String certURL)
	throws IOException, CertificateException, KeyStoreException,
	       KeyManagementException 
    {
	storeCertificate(certURL,false);
    }

    /**
     * method to retrieve the certificate from the CA after successful
     * authentication. It sends the full {@link CSR}, pemencoded. It expects as
     * reply a pem encoded certificate, which will be stored in the internal
     * {@link crypto.CryptoStore}. It uses the parameter <CODE>certURL</CODE> to
     * talk to and optionally sets the {@link javax.net.ssl.SSLSocketFactory}.
     * @param certURL URL to send the CSR to.
     * @param set <CODE>boolean</CODE> whether or not to set the
     * <CODE>SSLSocketFactory</CODE> to use the just downloaded certificate.
     * @throws IOException
     * @throws CertificateException
     * @throws KeyStoreException
     * @throws KeyManagementException when setting the
     * <CODE>SSLSocketFactory</CODE> fails 
     */
    public void storeCertificate(String certURL, boolean set)
	throws IOException, CertificateException, KeyStoreException,
	       KeyManagementException
    {
	CAConnection connection=null;
	int response=HttpURLConnection.HTTP_OK;
	X509Certificate x509Cert=null;
	boolean retry;

	CSR csr=cryptoStore.getCSR();
	if (csr==null)	{ // internal error, no use to retry
	    comm.error("Cannot find a Certificate Signing request in "+
			       "the CryptoStore",null);
	    throw new KeyStoreException("Empty CSR in CryptoStore");
	}

	// Start of communication, and hence retry loop...
	do {
	    retry=false; // reset here; only retry when needed...
	    try { // Open CAConnection to certURL
		connection=new CAConnection(certURL);
	    } catch (IOException e) {
		if (retry=comm.retry("Could not connect to\n"+
					     "<I>"+certURL+"</I>",e))
		    continue;
		throw e;
	    }
	    try { // Send CSR and catch response code
		response=connection.postString("csr="+csr.pemString());
	    } catch (IOException e)	{
		if (retry=comm.retry("Error during POST of pemstring to\n"+
					     "<I>"+certURL+"</I>",e))
		    continue;
		throw e;
	    }
	    // Check response code HTTP_OK (=200)
	    if (response!=HttpURLConnection.HTTP_OK)  {
		if (retry=comm.retry("Expected response "+
			    HttpURLConnection.HTTP_OK+
			    ", found:\n<I>"+connection.getResponseMessage()+
			    "</I>\nfrom:\n<I>"+certURL+"</I>",null))
		    continue;
		throw new IOException("Expected "+HttpURLConnection.HTTP_OK+
				      ", found "+response);
	    }
	    // Read the certificate
	    try {
		x509Cert=connection.getCert();
	    } catch (CertificateException e)  {
		final int maxLength=500;
		String responseString=connection.getResponse();
		if (responseString.length()>maxLength)
		    responseString=responseString.
					substring(0,maxLength-1)+
					"\n...[REST OF RESPONSE SUPPRESSED]...";
		if (retry=comm.retry("Server response from"+
			    "\n<I>"+certURL+
			    "</I>\nis not a valid certificate:\n<I>"+
			    responseString+
			    "</I>\nMake sure you have successfully logged in!",
			    null))
		    continue;
		throw e;
	    } catch (IOException e)	{
		if (retry=comm.retry("Error while downloading certificate from"+
					     "\n<I>"+certURL+"</I>",e))
		    continue;
		throw e;
	    }
	    // Check if it worked
	    if (x509Cert==null) {
		if (retry=comm.retry("Downloaded certificate is empty",null))
		    continue;
		throw new CertificateException("Downloaded certificate is empty");
	    }
	} while (retry); // End of retry loop...

	// Now store the certificate
	try {
	    cryptoStore.storeCertificate(x509Cert);
	    comm.success(x509Cert.getSubjectX500Principal().toString());
	} catch (Exception e)	{ // internal error, no use to retry...
	    comm.error("Cannot store certificate in CryptoStore",e);
	    throw new KeyStoreException(e.getMessage());
	}
	// optionally set the SSLSocketFactory
	if (set)  {
	    try	{ // NOTE: CAHttps knows which needs to set: https or httx
		setSSLSocketFactory();
	    } catch(Exception e)    { // internal error, no use to retry...
		comm.error("Cannot set the SSLSocketFactory",e);
		throw new KeyManagementException(e.getMessage());
	    }
	}
    }
   
    /**
     * Sets the default {@link SURFCACommunicator} to use for user interaction,
     * the actual communicator used for new instances of
     * <CODE>SURFCAHttps</CODE> can be overriden using the constructor {@link
     * #SURFCAHttps(SURFCACommunicator)}.
     * It checks whether the requested method is possible, otherwise it uses the
     * default <CODE>"stdio"</CODE>.
     * @param commInput <CODE>String</CODE> describing which type to use, valid
     * values are
     * <UL><LI><CODE>"stdio"</CODE> - use <CODE>stdio/stderr</CODE>
     * <LI><CODE>"popup"</CODE> - use (swing) popups
     * <LI><CODE>null</CODE> - use default <CODE>"stdio"</CODE></UL>
     * @return String describing the actual type being used.
     * @see #getCommunicator()
     */
    public static String setCommunicator(String commInput)  {
	final String defcomm="stdio";

	// If we don't have a gui, use stdio
	if (!GraphTools.isGraphic())
	    commString="stdio";
	else {
	    // If not specified, use default
	    if (commInput==null)
		commString=defcomm;
	    else
		commString=commInput.toLowerCase();
	}
	
	// Set the communicator
	if ("popup".equals(commString))	{
	    defaultComm=new SURFCAPopupComm();
	    return commString;
	} else if ("stdio".equals(commString))	{
	    defaultComm=new StdioComm();
	    return commString;
	} else { // Use default when unknown...
	    defaultComm=new StdioComm();
	    return commString;
	}
    }
    
    /**
     * Returns the default type of {@link SURFCACommunicator} used for user
     * interaction.
     * @return String describing the type being used.
     * @see #setCommunicator(String)
     */
    public static String getCommunicator()  {
	return commString;
    }

     /**
     * Setter method for the local private {@link #showSuccess}.
     * @param set <CODE>boolean</CODE> to put into private
     * <CODE>showSuccess</CODE> field.
     * @see CAHttps#SUCCESS_PROP
     * @see #getShowSuccess() */
    public static void setShowSuccess(boolean set)  {
	showSuccess=set;
    }
    
    /**
     * Getter method for the local private {@link #showSuccess}.
     * @return <CODE>boolean</CODE> value of private field
     * <CODE>showSuccess</CODE>.
     * @see CAHttps#SUCCESS_PROP
     * @see #setShowSuccess(boolean) */
    public static boolean getShowSuccess()  {
	return showSuccess;
    }

    /**
     * Interface for {@link SURFCAHttps} communication with the user.
     * @see StdioComm
     * @author Mischa Sall&eacute;
     * @version 0.1
     */ 
    public interface SURFCACommunicator	{
	/**
	 * Called just before the webbrowser is started.
	 * @throws IOException
	 */
	public void preBrowse() throws IOException;
	/**
	 * Called just after the webbrowser is started, which can be used to
	 * give us feedback when the user is ready with the webbrowser.
	 * @throws IOException
	 */
	public void postBrowse() throws IOException;
	/**
	 * Called when an error occurs. It prints an error string using the
	 * text and e.g. the {@link Exception#getMessage()} from
	 * <CODE>e</CODE>.
	 * @param text <CODE>String</CODE> - some text.
	 * @param e <CODE>Exception</CODE> that caused the error (if any).
	 * @see #retry(String,Exception)
	 */
	public void error(String text,Exception e);
	/**
	 * Identical to {@link #error(String,Exception)} except that it is
	 * called when an error occurs that might be fixed by the user. (S)He
	 * can then choose to retry.
	 * @param text <CODE>String</CODE> - some text.
	 * @param e <CODE>Exception</CODE> that caused the error (if any).
	 * @return boolean whether or not the user wishes to retry.
	 * @see #error(String,Exception)
	 */
	public boolean retry(String text,Exception e);
	/**
	 * Called when the interaction has been successful and the certificate
	 * stored. It is supposed to make use of {@link #getShowSuccess()} to
	 * decide whether to show anything.
	 * @param text <CODE>String</CODE> - some text
	 */
	public void success(String text);
    }
    
    /**
     * Implementation of a {@link SURFCACommunicator} using simple text via
     * stdin/stderr/stdout.
     * @see SURFCACommunicator
     */
    static class StdioComm implements SURFCACommunicator   {
	/**
	 * Called just before the webbrowser is started, prints a text and waits
	 * for confirmation.
	 * @throws IOException
	 */
	public void preBrowse()	throws IOException
	{
	    System.out.println("A new webbrowser or webbrowsertab is "+
			       "about to start.\n"+
			       "You will be asked to choose your "+
			       "Identity Provider and log in.\n"+
			       "1) When ready press ENTER and follow "+
			       "instructions in BROWSER...\n"+
			       "2) When DONE press enter AGAIN!!");
	    ConsoleTools.readLine();
	}
	/**
	 * Called just after the webbrowser is started, prints a text and waits
	 * for confirmation, effectively blocking execution until the user is
	 * ready with the webbrowser.
	 * @throws IOException
	 */
	public void postBrowse() throws IOException
	{
	    System.out.println("Browser started/called, waiting for second ENTER...");
	    ConsoleTools.readLine();
	}

	/**
	 * Called upon error. It prints an error string using the
	 * text and {@link Exception#getMessage()} from
	 * <CODE>e</CODE> (if non-null) on <CODE>stderr</CODE>, any HTML italics
	 * tags are removed.
	 * @param text <CODE>String</CODE> - some descriptive text.
	 * @param e <CODE>Exception</CODE> that caused the error (if any).
	 */
	public void error(String text,Exception e)    {
	    String textConverted=text.replaceAll("</*I>"," ");
	    System.err.println("Error: "+textConverted);
	    if (e!=null)
		System.err.println(" "+e.getMessage());
	}
	
	/**
	 * Identical to {@link #error(String,Exception)} except that it is called
	 * when an error occurs that might be fixed by the user. (S)He can then
	 * choose to retry.
	 * @param text <CODE>String</CODE> - some text.
	 * @param e <CODE>Exception</CODE> that caused the error (if any).
	 * @return boolean whether or not the user wishes to retry.
	 * @see #error(String,Exception)
	 */
	 public boolean retry(String text,Exception e)    {
	    error(text,e);
	    try	{
		return ConsoleTools.getConfirm("Retry");
	    } catch(IOException f)  {
		System.err.println("Caught exception: "+f.getMessage());
		return false;
	    }
	}
	
	/**
	 * It is called when interaction has been successful and the certificate
	 * stored, when {@link #getShowSuccess()} equals <CODE>true</CODE>, it
	 * will then print a confirmation.
	 * @param text <CODE>String</CODE> - text describing the certificate
	 * subject.
	 */
	public void success(String text)    {
	    if (SURFCAHttps.getShowSuccess())
		System.out.println("Successfully imported certificate for\n "+
				   text);
	}

    }
}
