package nl.nikhef.slcshttps;

import nl.nikhef.slcshttps.util.ConsoleTools;
import nl.nikhef.slcshttps.gui.GraphTools;
import nl.nikhef.slcshttps.gui.PKCS12PopupComm;

import java.security.KeyStore;
import java.io.File;
import java.io.FileInputStream;
import java.security.cert.X509Certificate;

import java.io.IOException;
import java.security.cert.CertificateException;
import java.security.KeyStoreException;
import java.security.KeyManagementException;
import java.security.NoSuchProviderException;

/**
 * Implementation of the <CODE>abstract</CODE> {@link CAHttps} for importing a
 * PKCS12 file from disk. Typical usage consists of calling
 * <UL><LI>{@link #initialize()} or {@link #initialize(String)}
 * <LI>{@link #storeCertificate()}
 * <LI>{@link #setHttpsSSLSocketFactory()} this can
 * be combined directly by calling <CODE>storeCertificate(true)</CODE>.
 * </UL>
 * @see CAHttps
 * @author Mischa Sall&eacute;
 * @version 0.1
 */
public class PKCS12Https extends CAHttps {
    /** Name of property defining which {@link PKCS12Communicator} to use:
     * {@value}. Valid values of this property are
     * <UL><LI><CODE>"stdio"</CODE> use <CODE>stdio/stderr</CODE>
     * <LI><CODE>"popup"</CODE> use (swing) popups</UL>
     * @see #setCommunicator(String)
     */
    private final static String COMMPROP="nl.nikhef.slcshttps.comm";
    /** filename of the PKCS12 file to be imported. */
    private String fileName=null;
    /** password for the PKCS12 file (both for the file and privatekey). */
    private char[] pkcs12Password=null;
    /** internal {@link KeyStore} which will hold the contents of the file. */
    private KeyStore pkcs12Store=null;

    /** Contains the the value of the property {@value COMMPROP}.
     * @see #getCommunicator() */
    private static String commString=null;
    /** The {@link PKCS12Communicator} to be used, can be set using {@link
     * #setCommunicator(String)}. */
    private static PKCS12Communicator comm=null;
    /** Initialize <CODE>commString</CODE> and <CODE>comm</CODE> */
    static {
	// Valid options: stdio, popup, make it static for the class...
	String commProp=System.getProperty(COMMPROP);
	setCommunicator(commProp);
    }
    /** Defines whether to acknowledge successful certificate import, download
     * etc. Valid options are <CODE>true</CODE> or <CODE>false</CODE>. Default
     * is same as the value in the superclass {@link CAHttps#showSuccess} which
     * in turn is set by {@link CAHttps#SUCCESS_PROP}.
     * @see #getShowSuccess()
     * @see #setShowSuccess(boolean) */ 
    private static boolean showSuccess=CAHttps.showSuccess;

    /**************************************************************************
     * METHODS
     *************************************************************************/

    /**
     * Constructs a default <CODE>PKCS12Https</CODE>. The constructor for the
     * super class {@link CAHttps#CAHttps(boolean)} is called with argument
     * <CODE>false</CODE> since we don't use a CSR.
     * @throws KeyStoreException when initialization failed.
     * @see CAHttps#CAHttps(boolean)
     */
    public PKCS12Https() throws KeyStoreException {
	super(false);
	try {
	    /* Use BouncyCastle, since standard Java gives uncomprehensible
	     * error messages */
	    pkcs12Store=KeyStore.getInstance("PKCS12","BC");
	} catch (NoSuchProviderException e)   {
	    throw new KeyStoreException("Cannot use BouncyCastle provider");
	}
    }

    /**
     * method to initialize the import: this consists of having the user pick a
     * file, entering the password and reading in the PKCS12 file.
     * @throws IOException upon error, including an unreadable file.
     * @see #initialize(String,String)
     */
    public void initialize() throws IOException {
	initialize(null,null);
    }

    /**
     * method to initialize the import: this consists of having the user pick a
     * file when <CODE>path</CODE> denotes a directory, entering the password
     * and reading in the PKCS12 file.
     * @param path either path to PKCS12 file or directory with respect to which
     * a file is to be chosen.
     * @throws IOException upon error, including an unreadable file.
     * @see #initialize(String,String)
     */
    public void initialize(String path) throws IOException {
	initialize(path,null);
    }

    /**
     * method to initialize the import: this consists of having the user pick a
     * file when <CODE>path</CODE> denotes a directory reading in the PKCS12
     * file using the specified password.
     * @param path either path to PKCS12 file or directory to start the file chooser in.
     * @param password password to unlock PKCS12 store.
     * @throws IOException upon error, including an unreadable file.
     */
    public void initialize(String path,String password) throws IOException {
	// Get the filename
	if (path==null || (new File(path)).isDirectory())   {
	    // Read from current/default or specified directory
	    try {
		fileName=comm.getFile(path);
	    } catch(IOException e)  {
		comm.error(path,e); // throws new IOException
	    }
	    if (fileName==null) { // cancelled
		try {
		    // Initialize an empty KeyStore...
		    pkcs12Store.load(null,null);
		    return;
		} catch (Exception e)	{
		    comm.error(fileName,e); // throws new IOException
		}
	    }
	} else
	    fileName=path;

	// Get the password
	if (password==null)
	    pkcs12Password=comm.getPassword("Enter PKCS12 password: ");
	else
	    pkcs12Password=password.toCharArray();

	// Load the file
	try {
	    pkcs12Store.load(new FileInputStream(fileName),pkcs12Password);
	} catch (Exception e)	{
	    comm.error(fileName,e); // throws new IOException
	}
    }

    /**
     * method to finalize the import: this consists of getting the
     * certificate/key from the <CODE>pkcs12Store</CODE> loaded in {@link
     * #initialize()} and putting it in the internal {@link crypto.CryptoStore}.
     * @throws IOException
     * @throws CertificateException
     * @throws KeyStoreException
     * @throws KeyManagementException never really thrown, since we don't set
     * the <CODE>SSLSocketFactory</CODE>.
     * @see #storeCertificate(String,boolean)
     */
    public void storeCertificate()
	throws IOException, CertificateException, KeyStoreException,
		KeyManagementException
    {
	storeCertificate(pkcs12Password,false);
    }

    /**
     * method to finalize the import: this consists of getting the
     * certificate/key from the <CODE>pkcs12Store</CODE> loaded in {@link
     * #initialize()} and putting it in the internal {@link crypto.CryptoStore};
     * it optionally sets the {@link javax.net.ssl.SSLSocketFactory}.
     * @throws IOException
     * @throws CertificateException
     * @throws KeyStoreException
     * @throws KeyManagementException when setting the
     * <CODE>SSLSocketFactory</CODE> fails
     * @param set <CODE>boolean</CODE> whether or not to set the
     * <CODE>SSLSocketFactory</CODE>.
     * @see #storeCertificate(String,boolean)
     */
    public void storeCertificate(boolean set)
	throws IOException, CertificateException, KeyStoreException,
		KeyManagementException
    {
	storeCertificate(pkcs12Password,set);
    }

    /**
     * method to finalize the import: this consists of getting the
     * certificate/key from the <CODE>pkcs12Store</CODE> loaded in {@link
     * #initialize()} and putting it in the
     * internal {@link crypto.CryptoStore}; uses <CODE>password</CODE> for the
     * import password.
     * @param password <CODE>String</CODE> representation of the <CODE>pkcs12Store</CODE>
     * password.
     * @throws IOException
     * @throws CertificateException
     * @throws KeyStoreException
     * @throws KeyManagementException never really thrown, since we don't set
     * the <CODE>SSLSocketFactory</CODE>.
     * @see #storeCertificate(String,boolean)
     */
    public void storeCertificate(String password)
	throws IOException, CertificateException, KeyStoreException,
		KeyManagementException
    {
	storeCertificate(password.toCharArray(),false);
    }

    /**
     * method to finalize the import: this consists of getting the
     * certificate/key from the <CODE>pkcs12Store</CODE> loaded in {@link
     * #initialize()} and putting it in the internal {@link crypto.CryptoStore}; it
     * optionally sets the {@link javax.net.ssl.SSLSocketFactory} and uses
     * <CODE>password</CODE> for the import password.
     * @param password <CODE>String</CODE> representation of the <CODE>pkcs12Store</CODE>
     * password.
     * @param set <CODE>boolean</CODE> whether or not to set the
     * <CODE>SSLSocketFactory</CODE>.
     * @throws IOException
     * @throws CertificateException
     * @throws KeyStoreException
     * @throws KeyManagementException when setting the
     * <CODE>SSLSocketFactory</CODE> fails
     */
    public void storeCertificate(String password,boolean set)
	throws IOException, CertificateException, KeyStoreException,
		KeyManagementException
    {
	storeCertificate(password.toCharArray(),set);
    }

    /**
     * method to finalize the import: this consists of getting the
     * certificate/key from the <CODE>pkcs12Store</CODE> loaded in {@link
     * #initialize()} and putting it in the internal {@link crypto.CryptoStore}; it
     * optionally sets the {@link javax.net.ssl.SSLSocketFactory} and uses
     * <CODE>passwordCharArr</CODE> for the import password.
     * @param passwordCharArr <CODE>char[]</CODE> representation of the
     * <CODE>pkcs12Store</CODE> password.
     * @param set <CODE>boolean</CODE> if set to <CODE>true</CODE> sets the
     * <CODE>SSLSocketFactory</CODE> to use the just downloaded certificateboolean
     * @throws IOException
     * @throws CertificateException
     * @throws KeyStoreException
     * @throws KeyManagementException when setting the
     * <CODE>SSLSocketFactory</CODE> fails
     * @see #storeCertificate(String,boolean)
     */
    public void storeCertificate(char[] passwordCharArr,boolean set)
	throws IOException, CertificateException, KeyStoreException, 
		KeyManagementException
    {	
	X509Certificate x509Cert=cryptoStore.importPKCS12(pkcs12Store,passwordCharArr);
	comm.success(fileName,x509Cert.getSubjectX500Principal().toString());
	// Now optionally set the SSLSocketFactory
	if (set)    {
	    try	{
		setSSLSocketFactory(); // NOTE: CAHttps knows which we need to set
	    } catch(Exception e)    {
		throw new KeyManagementException("Cannot set the SSLSocketFactory: "+
					    e.getMessage());
	    }
	}
    }

    /**
     * Sets the {@link PKCS12Communicator} to use for user interaction, it
     * checks whether the requested method is possible, otherwise use the
     * default.
     * @param commInput <CODE>String</CODE> describing which type to use, valid
     * values are
     * <UL><LI><CODE>"stdio"</CODE> - use <CODE>stdio/stderr</CODE>
     * <LI><CODE>"popup"</CODE> - use (swing) popups
     * <LI><CODE>null</CODE> - use default <CODE>"stdio"</CODE></UL>
     * @return String describing the actual type being used.
     * @see #getCommunicator()
     */
    public static String setCommunicator(String commInput)  {
	final String defcomm="stdio";

	// If we don't have a gui, use stdio
	if (!GraphTools.isGraphic())
	    commString="stdio";
	else {
	    // If not specified, use default
	    if (commInput==null)
		commString=defcomm;
	    else
		commString=commInput.toLowerCase();
	}

	// Set the communicator
	if ("popup".equals(commString))	{
	    comm=new PKCS12PopupComm();
	    return commString;
	} else if ("stdio".equals(commString))	{
	    comm=new StdioComm();
	    return commString;
	} else { // Use default when unknown...
	    comm=new StdioComm();
	    return commString;
	}
    }
   
    /**
     * Returns the type of {@link PKCS12Communicator} used for user interaction.
     * @return String describing the type being used.
     * @see #setCommunicator(String)
     */
    public static String getCommunicator()  {
	return commString;
    }
    
    /**
     * Setter method for the local private {@link #showSuccess}.
     * @param set <CODE>boolean</CODE> to put into private
     * <CODE>showSuccess</CODE> field.
     * @see CAHttps#SUCCESS_PROP
     * @see #getShowSuccess() */
    public static void setShowSuccess(boolean set)  {
	showSuccess=set;
    }
    
    /**
     * Getter method for the local private {@link #showSuccess}.
     * @return <CODE>boolean</CODE> value of private field <CODE>showSuccess</CODE>
     * @see CAHttps#SUCCESS_PROP
     * @see #setShowSuccess(boolean) */
    public static boolean getShowSuccess()  {
	return showSuccess;
    }

    /** Interface for {@link PKCS12Https} communication with the user.
     * @see StdioComm
     * @author Mischa Sall&eacute;
     * @version 0.1
     */
    public static interface PKCS12Communicator	{
	/**
	 * method to get the PKCS12 password from the user.
	 * @param prompt <CODE>String</CODE> to print before input.
	 * @return char[] representation of the password
	 * @throws IOException upon I/O error
	 */
	public char[] getPassword(String prompt) throws IOException;
	/**
	 * method to get the PKCS12 filename from the user.
	 * @param path <CODE>String</CODE> path to start in, use as offset. 
	 * @return String representation of the absolute filename.
	 * @throws IOException upon I/O error
	 */
	public String getFile(String path) throws IOException;
	/**
	 * called when an error occurs. It prints an error string using the
	 * filename and e.g. the {@link Exception#getMessage()} from
	 * <CODE>e</CODE>.
	 * @param filename <CODE>String</CODE> describing the PKCS12 filename
	 * which was tried (if any).
	 * @param e <CODE>Exception</CODE> that caused the error (if any).
	 * @throws IOException with a (new) description of the problem
	 */
	public void error(String filename,Exception e) throws IOException;
	/**
	 * called when the PKCS12 file is successfully imported and the
	 * certificate stored. It is supposed to make use of {@link
	 * #getShowSuccess()} to decide whether to show anything.
	 * @param filename <CODE>String</CODE> describing the PKCS12 filename
	 * which was imported.
	 * @param subject <CODE>String</CODE> describing the Subject DN of the
	 * certificate.
	 */
	public void success(String filename,String subject);
    }

    /**
     * This Implementation uses only <CODE>stdio/stderr</CODE> for I/O.
     * @author Mischa Sall&eacute;
     * @version 0.1
     * @see PKCS12Communicator
     */
    static class StdioComm implements PKCS12Https.PKCS12Communicator   {
	/**
	 * Method to get the PKCS12 password from the user, using {@link
	 * util.ConsoleTools#getPassword(String)}.
	 * @param prompt <CODE>String</CODE> to print before input.
	 * @return char[] representation of the password
	 * @throws IOException upon I/O error
	 */
	public char[] getPassword(String prompt) throws IOException    {
	    return ConsoleTools.getPassword(prompt);
	}

	/**
	 * Method to get the PKCS12 filename from the user, using
	 * <CODE>stdout</CODE> and {@link util.ConsoleTools#readLine()}.
	 * @param path <CODE>String</CODE> path to start in, use as offset. 
	 * @return String representation of the absolute filename.
	 * @throws IOException upon I/O error (including unreadability).
	 */
	public String getFile(String path) throws IOException  {
	    String fileName,pathName;
	    if (path==null) {
		File dir=new File(".");
		path=dir.getCanonicalPath()+File.separatorChar;
	    }
	    System.out.println("Give PKCS12 filename (with respect to "+path+")");
	    fileName=ConsoleTools.readLine();
	    if (fileName==null || fileName.length()==0) // Cancel...
		return null;
	    if (fileName.charAt(0)=='/')
		pathName=fileName;
	    else
		pathName=path+fileName;
	    File file=new File(pathName);
	    if (file.canRead())
		return pathName;
	    else
		throw new IOException("File "+pathName+" is unreadable");
	}

	/**
	 * Called when an error occurs, printing an error string about the
	 * filename (if non-null) and the {@link Exception#getMessage()} from
	 * <CODE>e</CODE> (when non-null) on <CODE>stderr</CODE>.
	 * @param filename <CODE>String</CODE> describing the PKCS12 filename
	 * which was tried.
	 * @param e <CODE>Exception</CODE> that caused the error (if any).
	 * @throws IOException with a (new) description of the problem
	 */
	public void error(String filename, Exception e) throws IOException  {
	    System.err.println("Error: Cannot import PKCS12 file");
	    if (filename!=null)
		System.err.println("\n "+filename);
	    throw new IOException("Cannot import PKCS12: "+e.getMessage());
	}

	/**
	 * Called when the PKCS12 file is successfully imported and the
	 * certificate stored; when {@link #getShowSuccess()} equals
	 * <CODE>true</CODE> it will print a confirmation.
	 * @param filename <CODE>String</CODE> describing the PKCS12 filename
	 * which was imported.
	 * @param subject <CODE>String</CODE> describing the Subject DN of the
	 * certificate.
	 */
	public void success(String filename,String subject)    {
	    if (PKCS12Https.getShowSuccess())
		System.out.println("Successfully imported PKCS12 file: "+
				filename+"\nSubject:\n "+subject);
	}
    }
}
