package nl.nikhef.slcshttps.util;

import java.lang.reflect.Method;
import java.net.URI;
import java.net.URL;
import java.lang.reflect.InvocationTargetException;

/**
 * Static class to provide a way to launch a proper default webbrowser from
 * Java. In JDK 1.6 onwards this is done using
 * <CODE>java.awt.Desktop.browse()</CODE>, for older JDK versions we provide
 * sensible defaults. Supported platforms are: Mac OS X, GNU/Linux, Unix, Windows.
 * @author Dem Pilafian
 * @author Willem van Engen (wvengen@nikhef.nl)
 * @author Mischa Sall&eacute; adapted for use by <CODE>{@link
 * nl.nikhef.slcshttps}</CODE>
 */
public class BareBonesBrowserLaunch {
    /** Browsers to try for Unix/Linux. */
    private static final String[] browsers={
	// Freedesktop, http://portland.freedesktop.org/xdg-utils-1.0/xdg-open.html
	"xdg-open",
	// Debian
	"sensible-browser",
	// Otherwise call browsers directly
	"firefox",
	"opera",
	"konqueror",
	"epiphany",
	"mozilla",
	"netscape"
    };
    /** String to identify Mac OS, it should start with {@value}. */
    private static final String OS_MAC="Mac OS";
    /** String to identify Windows, it should start with {@value}. */
    private static final String OS_WIN="Windows";
    /** error message used upon failure. */
    private static final String errMsg = "Error attempting to launch web browser";

    /**
     * Opens a webbrowser to a URL.
     * @param url url to open, type <CODE>URL</CODE>.
     * @throws Exception upon error.
     * @see #openURL(String)
     */
    public static void openURL(URL url) throws Exception {
	openURL(url.toExternalForm());
    }

    /**
     * opens a webbrowser to a URL, tries first the JDK 1.6
     * <CODE>java.awt.Desktop.getDesktop().browse(new URI(url))</CODE> method,
     * if that fails the action is platform dependent.
     * @param url url to open, type <CODE>String</CODE>
     * @throws Exception upon error.
     */
    public static void openURL(String url) throws Exception {
	// Try java desktop API first (new in Java 1.6)
	// basically: java.awt.Desktop.getDesktop().browse(new URI(url));
	try {
	    Class<?> desktop = Class.forName("java.awt.Desktop");
	    Method getDesktop = desktop.getDeclaredMethod("getDesktop", new Class[] {});
	    Object desktopInstance = getDesktop.invoke(null, new Object[] {});
	    Method browse = desktop.getDeclaredMethod("browse", new Class[] {URI.class});
	    URI uri = new URI(url);
	    browse.invoke(desktopInstance, new Object[] {uri});
	    return; // Successfully started browser
	} catch(InvocationTargetException e)	{
	    throw new Exception(e.getCause().getMessage());
	} catch(Exception e) {
	    // Other exception: warning, this includes the absence of java
	    // desktop API, hence has to be caught!
	}

	// Failed, resort to executing the browser manually
	String osName = System.getProperty("os.name");
	try {
	    if (osName.startsWith(OS_MAC)) {
		Class<?> fileMgr = Class.forName("com.apple.eio.FileManager");
		Method openURL = fileMgr.getDeclaredMethod("openURL",
			new Class[] {String.class});
		openURL.invoke(null, new Object[] {url});
	    }
	    else if (osName.startsWith(OS_WIN)) {
		String cmd = "rundll32 url.dll,FileProtocolHandler " + url;
		Runtime.getRuntime().exec(cmd);
	    } else { //assume Unix or Linux: try any of <CODE>browsers</CODE>
		String browser = null;
		for (int count = 0; count < browsers.length && browser == null; count++)
		{
		    if (Runtime.getRuntime().exec(
			    new String[] {"which", browsers[count]}).waitFor() == 0)
			browser = browsers[count];
		}
		if (browser == null) {
		    throw new Exception("Could not find web browser");
		} else { // Found a browser:
		    int returnval=0;
		    try {
			Process proc=Runtime.getRuntime().exec(new String[] {browser, url});
			try { // Now check if it didn't end in failure:
			    // wait short time to give it time to fail, if it
			    // started ok, we won't notice this delay...
			    Thread.sleep(500);
			    returnval=proc.exitValue();
			} catch(IllegalThreadStateException e) {
			    // Process is still running, hence assume it started ok...
			}
		    } catch (Exception e)   {
//			// Problem executing browser
			throw e;
		    }
		    // If we get here, we didn't get an exception
		    // now check the exit val.
		    if (returnval!=0)	{
			throw new Exception("Browser "+browser+
					    " returned with exitval "+returnval);
		    }
		}
	    }
	}
	catch (Exception e) { // something failed...
	    throw new Exception(errMsg+": "+e.getLocalizedMessage()); 
	}
    }

}
