package nl.nikhef.slcshttps.trust;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.ContentHandlerFactory;
import java.net.FileNameMap;
import java.net.ProtocolException;
import java.net.URL;
import java.security.Permission;
import java.security.Principal;
import java.security.cert.Certificate;
import java.util.List;
import java.util.Map;
import java.util.Date;
import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.HttpsURLConnection;
import java.net.HttpURLConnection;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.SSLPeerUnverifiedException;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.SSLSession;

import nl.nikhef.slcshttps.util.ConsoleTools;
import nl.nikhef.slcshttps.gui.GraphTools;
import nl.nikhef.slcshttps.gui.HttxClientPopupComm;

/**
 * This class extends a {@link HttpsURLConnection} that can use a {@link
 * TrustManager} that not only can check the certificate chain but also the
 * hostname against the server certificate.
 * More details can be found in the {@link nl.nikhef.slcshttps.trust package
 * description}.<BR>
 * In addition it can also warn the user when his/her client certificate is
 * about to expire or has already expired. Communication about this is handled
 * by an implementation of {@link HttxClientCommunicator}. An implementation
 * using just <CODE>stdio</CODE> is given by {@link StdioComm}. It can be
 * configured using the following system properties:<UL>
 * <LI>{@value #EXPIREPROP} - <CODE>true</CODE> or <CODE>false</CODE>, whether
 * to warn for (almost) expired client certificates. Can also be set using
 * {@link #setExpire(String)}.
 * <LI>{@value #EXPIRETIMEPROP} - time <EM>before</EM> expiry, in seconds, to
 * give a warning. Another warning will be given first time <EM>after</EM>
 * expiry. Can also be set using {@link #setExpireWarnTime(String)}.
 * <LI>{@value #COMMPROP} - type of communication, either <CODE>"stdio"</CODE>
 * or <CODE>"popup"</CODE>, when unset, <CODE>"stdio"</CODE> is used. Can also
 * be set using {@link #setCommunicator(String)}.
 * </UL>
 * <CODE>HttxURLConnection</CODE> is implemented as a <EM>delegate</EM> class
 * for <CODE>HttpsURLConnection</CODE> since that class is abstract while its
 * actual implementation is kept internal in the JDK API. Note that undocumented
 * overriding methods just call the corresponding overridden method of the
 * implementation, see the overridden methods for documentation.
 * @author Mischa Sall&eacute;
 * @version 0.1
 */
public class HttxURLConnection extends HttpsURLConnection {
    /** Masked {@link HttpURLConnection} or {@link HttpsURLConnection}. */
    private HttpURLConnection httpImpl=null;

    /** Default {@link SSLSocketFactory}, is initialized with an empty set of
     * {@link javax.net.ssl.KeyManager} and a {@link TrustManagerImpl}. */
    private static SSLSocketFactory defaultSSLSocketFactory;

    /** Whether we want warnings, set at class initialization using the
     * property {@value EXPIREPROP} or using {@link #setExpire(String)}. */
    private static boolean clientExpireWarn;
    /** Time in milliseconds before expiry to warn the user, use 0 for warning
     * only after actual expiry, set at class initialization using the
     * property {@value EXPIRETIMEPROP} or using {@link
     * #setExpireWarnTime(String)}. */
    private static long clientExpireWarnTime;
    /** Name of property that determines whether to warn: {@value}, default is
     * true, see also {@link #EXPIRETIMEPROP}, can be overridden using {@link
     * #setExpire(String)}. */
    private final static String EXPIREPROP="nl.nikhef.slcshttps.httxclientwarn";
    /** Name of property that determines <EM>when</EM> to give an
     * 'about-to-expire' warning: {@value}, value is number of seconds; when it
     * is unspecified or invalid, 0 is assumed and only a warning upon actual
     * expiry is given, see also {@link #EXPIREPROP}, can be overridden using
     * {@link #setExpireWarnTime(String)}. */
    private final static String EXPIRETIMEPROP="nl.nikhef.slcshttps.httxclientwarntime";
    /** Name of property that sets the type of communicator: {@value}, can be
     * overridden using {@link #setCommunicator(String)}. */
    private final static String COMMPROP="nl.nikhef.slcshttps.comm";
    
    /** Expiry time of client side certificate (<CODE>null</CODE> for unset),
     * can be set using {@link #setClientExpireDate(Date)}. */
    private static Date clientExpireDate=null;
    /** Whether a warning has been given about imminent expiry. */
    private static boolean clientExpireWarnAcknowl;
    /** Whether a warning has been given about expiry. */
    private static boolean clientExpireAcknowl;

    /** describes the type of communicator in use, initialized using the value
     * of property {@value COMMPROP} by {@link #setCommunicator(String)}. */
    private static String commString=null;
    /** The {@link HttxClientCommunicator} used for communication about client
     * certificate expiry, can be set using {@link #setCommunicator(String)}. */
    private static HttxClientCommunicator comm=null;
    /** Initialize <CODE>commString</CODE> and <CODE>comm</CODE>. */
    static {
	// Valid options: stdio, popup, make it static for the class...
	String commProp=System.getProperty(COMMPROP);
	setCommunicator(commProp);
	// Set expiry warnings?
	String expireProp=System.getProperty(EXPIREPROP);
	setExpire(expireProp);
	// Set expiry early warnings and when?
	String expireTime=System.getProperty(EXPIRETIMEPROP);
	setExpireWarnTime(expireTime);
    }

    /** Default {@link HostnameVerifier}, is initialized to a trivial one, since
     * all functionality is in the {@link TrustManagerImpl}. */
    private static HostnameVerifier hostVerifier;
    static {
	// Set default SSLSocketFactory to use TrustManagerImpl
	SSLSocketFactory socketFactory=null;
	try {
	    SSLContext sslContext=SSLContext.getInstance("TLS");
	    sslContext.init(null,new TrustManager[]{new TrustManagerImpl()},null);
	    defaultSSLSocketFactory=sslContext.getSocketFactory();
	} catch (Exception e)	{
	    throw new RuntimeException(
		"Cannot get a SocketFactory for given managers",e);
	}
	
	// Set default HostnameVerifier to be trivial
	hostVerifier=new HostnameVerifier() {
		public boolean verify(String hostname, SSLSession session) {
		    return true;
		}
	    };
    }
    /** {@link URL} of the connection. */
    private URL url;
    /** hostname part of the {@link URL}. */
    private String hostName;
    /** portnumber of the {@link URL}. */
    private int portNumber;
    /** whether this is a HTTPS or HTTP connection. */
    private boolean ssl;

    /**************************************************************************
     * METHODS
     *************************************************************************/

    /**
     * Constructs a <CODE>HttxURLConnection</CODE> from given {@link
     * HttpURLConnection} or {@link HttpsURLConnection} (a subclass of the
     * former). If this is a SSL connection (i.e. Https) it sets the hostname
     * and portnumber of the URL into the static {@link TrustManagerImpl}
     * fields. The preferred way is to call the static method {@link
     * #openConnection(URL)}. We like to have the constructor public instead of
     * protected: we do not have the option to <EM>cast</EM> the object
     * returned by {@link URL#openConnection()} since that will be a JDK
     * internal implementation of e.g. <CODE>HttpsURLConnection</CODE>; however,
     * we can use the constructor to construct a new
     * <CODE>HttxURLConnection</CODE> from such an object.
     * @param httpConnection <CODE>HttpURLConnection</CODE> to construct a
     * <CODE>HttxURLConnection from.
     * @throws IOException in case of error, including expired client
     * certificate.
     * @see #openConnection(URL)
     */
    public HttxURLConnection(HttpURLConnection httpConnection)
	throws IOException
    {
	super(null);

	httpImpl=httpConnection;
	url=getURL();
	if ("https".equals(url.getProtocol()))    {
	    ssl=true;
	    // Set SSLSocketFactory of this instance, not of whole class, to
	    // stay compatible with HttpsURLConnection.
	    setSSLSocketFactory(defaultSSLSocketFactory);
	    // Set the HostnameVerifier of the open connection
	    ((HttpsURLConnection)httpImpl).setHostnameVerifier(hostVerifier);
	    // Set host and portnumber
	    hostName=url.getHost();
	    portNumber=url.getPort();
	    if (portNumber==-1) portNumber=url.getDefaultPort();
	    TrustManagerImpl.setHostname(hostName);
	    TrustManagerImpl.setPort(portNumber);
	    // Check if client cert hasn't expired
	    if (!checkClientExpire())	{
		throw new IOException("Client Certificate has expired");
	    }
	} else
	    ssl=false;
    }

    /**
     * Constructs a <CODE>HttxURLConnection</CODE> to <CODE>url</CODE>.
     * The preferred way is to call the static method
     * {@link #openConnection(URL)}.
     * @param url <CODE>URL</CODE> to open a connection to.
     * @throws IOException
     * @see #HttxURLConnection(HttpURLConnection)
     * @see #openConnection(URL)
     */
    public HttxURLConnection(URL url) throws IOException    {
	// Cast to HttpURLConnection since that's good for Https and Http
	this((HttpURLConnection)url.openConnection());
    }

    /**
     * Creates and returns a new <CODE>HttxURLConnection</CODE> for given
     * <CODE>url</CODE>. This is the preferred method to open the connection.
     * @param url <CODE>URL</CODE> to open a connection to.
     * @return HttpxURLConnection to <CODE>url</CODE>
     * @throws IOException upon error
     */
    public static HttxURLConnection openConnection(URL url) 
	throws IOException
    {
	return new HttxURLConnection(url);
    }
 
    /**
     * Connects an opened <CODE>HttxURLConnection</CODE>. Note that it is not
     * necessary to explicitly call {@link HttpsURLConnection#connect()}, it
     * will automatically connect when starting I/O to it. This method will set
     * (again) the hostname and portnumber of the {@link TrustManagerImpl},
     * which will not happen when it is connected automatically.
     * @throws IOException upon error
     */
    public void connect() throws IOException {
	if (ssl)    {
	    // Set hostname/port again, in case they have changed between
	    // openConnection and connect. Note however that we also can read
	    // without doing a connect() E.g. using connection.getInputStream()
	    TrustManagerImpl.setHostname(hostName);
	    TrustManagerImpl.setPort(portNumber);
	}
        httpImpl.connect();
    }

    /**
     * Disconnects an opened <CODE>HttxURLConnection</CODE>, which also resets
     * the hostname and portnumber for the {@link TrustManagerImpl}.
     */
    public void disconnect() {
        httpImpl.disconnect();
	TrustManagerImpl.setHostname(null);
	TrustManagerImpl.setPort(1);
    }
   
    /**
     * Sets the default {@link SSLSocketFactory} for
     * <CODE>HttxURLConnection</CODE>. Note that this does <EM>not</EM> set the
     * default <CODE>SSLSocketFactory</CODE> for {@link HttpsURLConnection}.
     * @param sslSocketFactory will be set as default.
     * @see #getDefaultSSLSocketFactory()
     */
    public static void setDefaultSSLSocketFactory(SSLSocketFactory sslSocketFactory)
    {
	defaultSSLSocketFactory=sslSocketFactory;
    }
   
    /**
     * Method to get the default {@link SSLSocketFactory} for
     * <CODE>HttxURLConnection</CODE>. Note that this is different from the
     * default <CODE>SSLSocketFactory</CODE> for {@link HttpsURLConnection}.
     * @return default <CODE>SSLSocketFactory</CODE>
     * @see #setDefaultSSLSocketFactory(SSLSocketFactory)
     */
    public static SSLSocketFactory getDefaultSSLSocketFactory() {
        return defaultSSLSocketFactory;
    }

    /**
     * Empty setter method, in order to override the method.
     * @param hostnameVerifier ignored.
     * @see #getHostnameVerifier().
     */
    public void setHostnameVerifier(HostnameVerifier hostnameVerifier) {
    }

    /**
     * Empty setter method, in order to override the method.
     * @param hostnameVerifier ignored.
     * @see #getDefaultHostnameVerifier().
     */
    public static void setDefaultHostnameVerifier(HostnameVerifier hostnameVerifier)
    {
    }

    /**
     * Method to get the {@link HostnameVerifier} for
     * <CODE>HttxURLConnection</CODE>. This is actually the same as returned by
     * {@link #getDefaultHostnameVerifier()} since we do not implement setter
     * methods.
     * @return HostnameVerifier used by <CODE>HttxURLConnection</CODE>.
     * @see #getDefaultHostnameVerifier().
     * @see #setHostnameVerifier(HostnameVerifier).
     */
    public HostnameVerifier getHostnameVerifier() {
        return hostVerifier;
    }

    /**
     * Method to get the default {@link HostnameVerifier} for
     * <CODE>HttxURLConnection</CODE>. This is actually the same as returned by
     * {@link #getHostnameVerifier()} since we do not implement setter methods.
     * @return HostnameVerifier used by <CODE>HttxURLConnection</CODE>.
     * @see #getDefaultHostnameVerifier().
     * @see #setDefaultHostnameVerifier(HostnameVerifier).
     */
    public static HostnameVerifier getDefaultHostnameVerifier() {
        return hostVerifier;
    }

    /**************************************************************************
     * METHODS for checking client certificate, mainly getters/setters, except
     * for checkClientExpire(), which does the actual checking.
     *************************************************************************/

    /**
     * Static method checking the client certificate, using status flags to
     * determine whether to warn the user the certificate expiry is imminent, is
     * already expired etc.
     * @return boolean whether we should continue.
     */
    public static boolean checkClientExpire()  {
	// Do we want warnings and is there a client cert present?
	if (!clientExpireWarn || clientExpireDate==null)
	    return true;

	// Get current msec
	long now=new Date().getTime();
	long exp=clientExpireDate.getTime();
	// Are we beyond expiry?
	if (now>exp)	{
	    // Have we already warned?
	    if (!clientExpireAcknowl)	{
		// Need warning with confirmation
		clientExpireAcknowl=true;
		return comm.expired(
		    "Your certificate has <I>expired</I>, "+
		    "do you still want to continue?");
	    }
	} else	{
	    if (clientExpireWarnTime>0 &&
		   now>exp-clientExpireWarnTime &&
		   !clientExpireWarnAcknowl)  {
		// Need warning without confirmation (it's still ok)
		clientExpireWarnAcknowl=true;
		comm.almostexpired("Your certificate will <I>expire</I>: "+
				    clientExpireDate.toString());
		return true;
	    }
	}
	return true;
    }

    /**
     * Method to set the expiry {@link Date} of the client side certificate,
     * <CODE>null</CODE> for no certificate. This also resets the acknowledgment
     * flags.
     * @param date expiry date of client certificate.
     * @see #getClientExpireDate()
     */
    public static void setClientExpireDate(Date date)	{
	clientExpireDate=date;
	// Make sure to reset acknowledgment flags
	clientExpireWarnAcknowl=false;
	clientExpireAcknowl=false;
    }

    /**
     * Method to get the expiry {@link Date} of the client side certificate,
     * <CODE>null</CODE> for no certificate.
     * @return expiry date of client certificate.
     * @see #setClientExpireDate(Date)
     */
    public static Date getClientExpireDate()	{
	return clientExpireDate;
    }

    /**
     * Sets whether to check for client certificate expiry and warn about it.
     * @param setString <CODE>String</CODE> representation of a boolean, whether
     * to check and warn about expiry of client certificate. If
     * <CODE>null</CODE> or invalid, use the default <CODE>true</CODE>.
     * @see #getExpire()
     * @see #EXPIREPROP
     */
    public static void setExpire(String setString)	{
	// when valid and equal false, use that
	if (setString!=null && setString.equals("false"))
	    clientExpireWarn=false;
	else
	    // Default
	    clientExpireWarn=true;
    }

    /**
     * Returns whether to check for client certificate expiry and warn about it.
     * @return boolean indicating whether to check and warn.
     * @see #setExpire(String)
     * @see #EXPIREPROP
     */
    public static boolean getExpire()	{
	return clientExpireWarn;
    }

    /**
     * Sets the time <EM>before</EM> expiry of a client certificate to give a
     * warning. When <CODE>timeString</CODE> is <CODE>null</CODE> or is not a
     * valid number (which includes negative numbers), it will be set to 0,
     * meaning no special 'about-to-expire' warning will be given, only a
     * 'already-expired'.
     * @param timeString <CODE>String</CODE> representation of the time in
     * seconds before expiry time when a warning should be given.
     * @see #getExpireWarnTime()
     * @see #EXPIRETIMEPROP
     */
    public static void setExpireWarnTime(String timeString) {
	// When empty, use default.
	if (timeString==null)
	    clientExpireWarnTime=0;
	else	{
	    try {
		// property is time in seconds, we need milliseconds.
		clientExpireWarnTime=Long.parseLong(timeString.trim())*1000;
		if (clientExpireWarnTime<0)
		    clientExpireWarnTime=0;
	    } catch (NumberFormatException nfe) {
		// When invalid, use default.
		clientExpireWarnTime=0;
	    }
	}
    }

    /**
     * Returns the number of milliseconds before actual expiry when a warning
     * about imminent expiry of the client certificate is given.
     * @return number of milliseconds before expiry time.
     * @see #setExpireWarnTime(String)
     * @see #EXPIRETIMEPROP
     */
    public static long getExpireWarnTime()  {
	return clientExpireWarnTime;
    }

    /**
     * Sets the type of {@link HttxClientCommunicator} based on
     * <CODE>commInput</CODE>. Valid values are:<UL>
     * <LI><CODE>"stdio"</CODE> - use <CODE>stdin/stdout/stderr</CODE>
     * <LI><CODE>"popup"</CODE> - use (swing) popups
     * <LI><CODE>null</CODE> - use default <CODE>"stdio"</CODE>
     * </UL>
     * @param commInput <CODE>String</CODE> describing the wished type of
     * communicator to be used.
     * @return <CODE>String</CODE> describing the actual type being used.
     * @see #getCommunicator()
     */
    public static String setCommunicator(String commInput)  {
	final String defcomm="stdio";

	// If we don't have a gui, use stdio
	if (!GraphTools.isGraphic())
	    commString="stdio";
	else {
	    // If not specified, use default
	    if (commInput==null)
		commString=defcomm;
	    else
		commString=commInput.toLowerCase();
	}
	
	// Set the communicator
	if ("popup".equals(commString))	{
	    comm=new HttxClientPopupComm();
	    return commString;
	} else if ("stdio".equals(commString))	{
	    comm=new StdioComm();
	    return commString;
	} else { // Use default when unknown...
	    comm=new StdioComm();
	    return commString;
	}
    }
    
    /**
     * Returns the type of {@link HttxClientCommunicator} used for user
     * interaction.
     * @return String describing the type being used.
     * @see #setCommunicator(String)
     */
    public static String getCommunicator()  {
	return commString;
    }

 

////////////////////////////////////////////////////////////////////////
// Below are only default methods which just shadow those in
// (Http(s))URLConnection, except for the final methods, which cannot be
// overriden
////////////////////////////////////////////////////////////////////////

   
////////////////////////////////////////////////////////////////////////
// Next set of methods are HttpsURLConnection only, hence they have to
// explicitly be cast into HttpsURLConnection or we get an error when httpsImpl
// is a HttpURLConnection.
////////////////////////////////////////////////////////////////////////

    public void setSSLSocketFactory(SSLSocketFactory arg0) {
        ((HttpsURLConnection)httpImpl).setSSLSocketFactory(arg0);
    }

    public Certificate[] getServerCertificates() throws SSLPeerUnverifiedException {
        return ((HttpsURLConnection)httpImpl).getServerCertificates();
    }

    public SSLSocketFactory getSSLSocketFactory() {
        return ((HttpsURLConnection)httpImpl).getSSLSocketFactory();
    }

    public Principal getPeerPrincipal() throws SSLPeerUnverifiedException {
        return ((HttpsURLConnection)httpImpl).getPeerPrincipal();
    }

    public Principal getLocalPrincipal() {
        return ((HttpsURLConnection)httpImpl).getLocalPrincipal();
    }

    public Certificate[] getLocalCertificates() {
        return ((HttpsURLConnection)httpImpl).getLocalCertificates();
    }

    public String getCipherSuite() {
        return ((HttpsURLConnection)httpImpl).getCipherSuite();
    }

////////////////////////////////////////////////////////////////////////
// The remainder are methods non-overridden methods.
// First the static ones, which call the HttpsURLConnection class methods
////////////////////////////////////////////////////////////////////////

    /**
     * Calls {@link HttpsURLConnection#setFileNameMap(FileNameMap)}.
     * @param arg0 <CODE>FileNameMap</CODE>
     * @see #getFileNameMap()
     * @see HttpsURLConnection#setFileNameMap(FileNameMap)
     */
    public static void setFileNameMap(FileNameMap arg0) {
        HttpsURLConnection.setFileNameMap(arg0);
    }

    /**
     * Calls {@link HttpsURLConnection#getFileNameMap()}.
     * @return FileNameMap
     * @see #setFileNameMap(FileNameMap)
     * @see HttpsURLConnection#getFileNameMap()
     */
    public static synchronized FileNameMap getFileNameMap() {
        return HttpsURLConnection.getFileNameMap();
    }

    /**
     * Calls {@link HttpsURLConnection#setDefaultAllowUserInteraction(boolean)}.
     * @param arg0 <CODE>boolean</CODE>
     * @see #getDefaultAllowUserInteraction()
     * @see HttpsURLConnection#setDefaultAllowUserInteraction(boolean)
     */
    public static void setDefaultAllowUserInteraction(boolean arg0) {
        HttpsURLConnection.setDefaultAllowUserInteraction(arg0);
    }

    /**
     * Calls {@link HttpsURLConnection#getDefaultAllowUserInteraction()}.
     * @return boolean containing default user interaction.
     * @see #setDefaultAllowUserInteraction(boolean)
     * @see HttpsURLConnection#getDefaultAllowUserInteraction()
     */
    public static boolean getDefaultAllowUserInteraction() {
        return HttpsURLConnection.getDefaultAllowUserInteraction();
    }

    /**
     * Calls {@link HttpsURLConnection#setFollowRedirects(boolean)}.
     * @param arg0 boolean
     * @see #getFollowRedirects()
     * @see HttpsURLConnection#setFollowRedirects(boolean)
     */
    public static void setFollowRedirects(boolean arg0) {
        HttpsURLConnection.setFollowRedirects(arg0);
    }

    /**
     * Calls {@link HttpsURLConnection#getFollowRedirects()}.
     * @return boolean with current follow redirects setting.
     * @see #setFollowRedirects(boolean)
     * @see HttpsURLConnection#getFollowRedirects()
     */
    public static boolean getFollowRedirects() {
        return HttpsURLConnection.getFollowRedirects();
    }

    /**
     * Calls {@link HttpsURLConnection#setContentHandlerFactory(ContentHandlerFactory)}.
     * @param arg0 <CODE>ContentHandlerFactory</CODE>
     * @see HttpsURLConnection#setContentHandlerFactory(ContentHandlerFactory)
     */
    public static synchronized void setContentHandlerFactory(ContentHandlerFactory arg0) {
        HttpsURLConnection.setContentHandlerFactory(arg0);
    }

    /**
     * Calls {@link HttpsURLConnection#guessContentTypeFromStream(InputStream)}.
     * @param arg0 <CODE>InputStream</CODE>
     * @return String containing content-type
     * @see HttpsURLConnection#guessContentTypeFromStream(InputStream)
     * @throws IOException
     */
    public static String guessContentTypeFromStream(InputStream arg0) throws IOException {
        return HttpsURLConnection.guessContentTypeFromStream(arg0);
    }

    /**
     * Calls {@link HttpsURLConnection#guessContentTypeFromName(String)}.
     * @param arg0 <CODE>String</CODE> with name.
     * @return String containing content-type
     * @see HttpsURLConnection#guessContentTypeFromName(String)
     */
    public static String guessContentTypeFromName(String arg0) {
        return HttpsURLConnection.guessContentTypeFromName(arg0);
    }

////////////////////////////////////////////////////////////////////////

    public int hashCode() {
        return httpImpl.hashCode();
    }

    public boolean equals(Object arg0) {
        return httpImpl.equals(arg0);
    }

    public String toString() {
        return httpImpl.toString();
    }

    public void setUseCaches(boolean arg0) {
        httpImpl.setUseCaches(arg0);
    }

    public void setRequestProperty(String arg0, String arg1) {
        httpImpl.setRequestProperty(arg0, arg1);
    }

    public void setReadTimeout(int arg0) {
        httpImpl.setReadTimeout(arg0);
    }

    public void setIfModifiedSince(long arg0) {
        httpImpl.setIfModifiedSince(arg0);
    }

    public void setDoOutput(boolean arg0) {
        httpImpl.setDoOutput(arg0);
    }

    public void setDoInput(boolean arg0) {
        httpImpl.setDoInput(arg0);
    }

    public void setDefaultUseCaches(boolean arg0) {
        httpImpl.setDefaultUseCaches(arg0);
    }

    public void setConnectTimeout(int arg0) {
        httpImpl.setConnectTimeout(arg0);
    }

    public void setAllowUserInteraction(boolean arg0) {
        httpImpl.setAllowUserInteraction(arg0);
    }

    public boolean getUseCaches() {
        return httpImpl.getUseCaches();
    }

    public URL getURL() {
        return httpImpl.getURL();
    }

    public String getRequestProperty(String arg0) {
        return httpImpl.getRequestProperty(arg0);
    }

    public Map<String, List<String>> getRequestProperties() {
        return httpImpl.getRequestProperties();
    }

    public int getReadTimeout() {
        return httpImpl.getReadTimeout();
    }

    public OutputStream getOutputStream() throws IOException {
        return httpImpl.getOutputStream();
    }

    public long getLastModified() {
        return httpImpl.getLastModified();
    }

    public InputStream getInputStream() throws IOException {
        return httpImpl.getInputStream();
    }

    public long getIfModifiedSince() {
        return httpImpl.getIfModifiedSince();
    }

    public Map<String, List<String>> getHeaderFields() {
        return httpImpl.getHeaderFields();
    }

    public int getHeaderFieldInt(String arg0, int arg1) {
        return httpImpl.getHeaderFieldInt(arg0, arg1);
    }

    public String getHeaderField(String arg0) {
        return httpImpl.getHeaderField(arg0);
    }

    public long getExpiration() {
        return httpImpl.getExpiration();
    }

    public boolean getDoOutput() {
        return httpImpl.getDoOutput();
    }

    public boolean getDoInput() {
        return httpImpl.getDoInput();
    }

    public boolean getDefaultUseCaches() {
        return httpImpl.getDefaultUseCaches();
    }

    public long getDate() {
        return httpImpl.getDate();
    }

    public String getContentType() {
        return httpImpl.getContentType();
    }

    public int getContentLength() {
        return httpImpl.getContentLength();
    }

    public String getContentEncoding() {
        return httpImpl.getContentEncoding();
    }

    public Object getContent(Class[] arg0) throws IOException {
        return httpImpl.getContent(arg0);
    }

    public Object getContent() throws IOException {
        return httpImpl.getContent();
    }

    public int getConnectTimeout() {
        return httpImpl.getConnectTimeout();
    }

    public boolean getAllowUserInteraction() {
        return httpImpl.getAllowUserInteraction();
    }

    public void addRequestProperty(String arg0, String arg1) {
        httpImpl.addRequestProperty(arg0, arg1);
    }

    public boolean usingProxy() {
        return httpImpl.usingProxy();
    }

    public void setRequestMethod(String arg0) throws ProtocolException {
        httpImpl.setRequestMethod(arg0);
    }

    public void setInstanceFollowRedirects(boolean arg0) {
        httpImpl.setInstanceFollowRedirects(arg0);
    }

    public void setFixedLengthStreamingMode(int arg0) {
        httpImpl.setFixedLengthStreamingMode(arg0);
    }

    public void setChunkedStreamingMode(int arg0) {
        httpImpl.setChunkedStreamingMode(arg0);
    }

    public String getResponseMessage() throws IOException {
        return httpImpl.getResponseMessage();
    }

    public int getResponseCode() throws IOException {
        return httpImpl.getResponseCode();
    }

    public String getRequestMethod() {
        return httpImpl.getRequestMethod();
    }

    public Permission getPermission() throws IOException {
        return httpImpl.getPermission();
    }

    public boolean getInstanceFollowRedirects() {
        return httpImpl.getInstanceFollowRedirects();
    }

    public String getHeaderFieldKey(int arg0) {
        return httpImpl.getHeaderFieldKey(arg0);
    }

    public long getHeaderFieldDate(String arg0, long arg1) {
        return httpImpl.getHeaderFieldDate(arg0, arg1);
    }

    public String getHeaderField(int arg0) {
        return httpImpl.getHeaderField(arg0);
    }

    public InputStream getErrorStream() {
        return httpImpl.getErrorStream();
    }

    /**
     * Interface for {@link HttxURLConnection} communication about client
     * certificate expiration.
     * @author Mischa Sall&eacute;
     * @version 0.1
     */
    public interface HttxClientCommunicator {
	/**
	 * Method to ask the user for confirmation to continue since the client
	 * certificate has expired.
	 * @param mesg <CODE>String</CODE> containing a message describing the
	 * problem.
	 * @return boolean <CODE>true</CODE> when the user wants to continue.
	 * <CODE>false</CODE> otherwise
	 */
	public boolean expired(String mesg);

	/**
	 * Method to inform the user that his client certificate is about to
	 * expire.
	 * @param mesg <CODE>String</CODE> containing a message describing the
	 * problem.
	 */
	public void almostexpired(String mesg);
    }
    
    /**
     * This Implementation uses only <CODE>stdio/stderr</CODE> for I/O.
     * @author Mischa Sall&eacute;
     * @version 0.1
     * @see HttxClientCommunicator
     */
    static class StdioComm implements HttxClientCommunicator    {
	/**
	 * Method to ask the user for confirmation to continue since the client
	 * certificate has expired, using stdio/stderr, any HTML italics tags
	 * are removed.
	 * @param mesg <CODE>String</CODE> containing a message describing the
	 * problem.
	 * @return boolean <CODE>true</CODE> when the user wants to continue,
	 * <CODE>false</CODE> otherwise.
	 */
	public boolean expired(String mesg)   {
	    String buffer="WARNING:\n "+mesg.replaceAll("</*I>"," ");
	    try {
		return ConsoleTools.getConfirm(buffer);
	    } catch(IOException e)	{
		System.err.println("Caught exception: "+e.getMessage());
		System.err.println("Not confirming validation");
		return false;
	    }
	}

	/**
	 * Method to inform the user that his client certificate is about to
	 * expire using stdio/stderr, any HTML italics tags are removed.
	 * @param mesg <CODE>String</CODE> containing a message describing the
	 * problem.
	 */
	public void almostexpired(String mesg)    {
	    String buffer=mesg.replaceAll("</*I>"," ");
	    System.err.println("WARNING:\n "+buffer);
	}
    }

}
