package nl.nikhef.slcshttps.trust;

import org.bouncycastle.jce.provider.BouncyCastleProvider;
import java.security.Security;

import java.security.cert.CertificateFactory;
import java.security.cert.CertPathValidator;
import java.security.cert.PKIXParameters;
import java.security.cert.PKIXBuilderParameters;
import java.security.cert.X509CertSelector;
import java.security.cert.CertPath;
import java.security.cert.X509Certificate;
import java.security.KeyStore;

import java.util.Date;
import java.util.Enumeration;
import java.util.Vector;

import java.io.File;
import java.io.FileInputStream;

import java.util.HashMap;
import java.util.Arrays;

import java.security.AccessController;
import java.security.PrivilegedExceptionAction;

import javax.security.auth.x500.X500Principal;

import java.security.KeyStoreException;
import java.security.cert.CertificateException;
import java.security.cert.CertPathValidatorException;
import java.io.FileNotFoundException;
import java.security.PrivilegedActionException;

//import org.bouncycastle.x509.PKIXCertPathReviewer;

/**
 * Static class to validate a {@link X509Certificate} chain. It provides only
 * one public (and static) method, {@link #validate(X509Certificate[])}. The
 * class is static for performance reasons, it's only initialized at startup.
 * It uses either the Java truststore specified via the default security
 * settings in $JAVA_HOME/lib/security/ or via an external truststore which can
 * be specified by defining the system property
 * <CODE>javax.net.ssl.trustStore</CODE>.
 * See <A HREF="http://java.sun.com/javase/6/docs/technotes/guides/security/jsse/JSSERefGuide.html#X509TrustManager">JSSE
 * Reference Guide</A> for details.
 * The code to find the correct truststore is roughly taken over from
 * the implementation dependent JDK1.6 class
 * <CODE>sun.security.ssl.TrustManagerFactoryImpl</CODE>.
 * The {@link CertPathValidator} uses BouncyCastle for stability and more human
 * readable error messages. BouncyCastle follows mostly <A
 * HREF="http://www.ietf.org/rfc/rfc3280.txt">RFC3280</A> which is now
 * superseded by <A HREF="http://www.ietf.org/rfc/rfc5280.txt">RFC5280</A>.
 * @author Mischa Sall&eacute;
 * @version 0.1
 */
public class CertChainChecker	{
    /** <CODE>KeyStore</CODE> with trusted certificates, initialized at class
     * initialization using {@link #getCacertsKeyStore()}. */
    private static KeyStore trustStore=getCacertsKeyStore();

    /** <CODE>certValidator</CODE> is doing the actual validation, initialized
     * at startup using {@link #initCertValidator()}. */
    private static CertPathValidator certValidator=initCertValidator();
   
    /** <CODE>certFactory</CODE> is needed to make a {@link CertPath} object
     * from a <CODE>{@link X509Certificate}[]</CODE>. */
    private static CertificateFactory certFactory=initCertFactory();

    /** <CODE>PKIXParameters</CODE> used by the {@link #certValidator}, they use
     * the {@link #trustStore} and are initialized at startup using {@link
     * #initPKIXParameters()}. */
    private static PKIXBuilderParameters pkixParameters=initPKIXParameters();

    /**
     * Validates a {@link X509Certificate} chain.
     * @param x509Chain array of certificates, they should be in the right
     * order.
     * @throws CertPathValidatorException when chain validation fails.
     * {@link Exception#getMessage()} gives the reason.
     * @throws CertificateException upon other error.
     * @see #validate(X509Certificate[],Date)
     */
    public static void validate(X509Certificate[] x509Chain)
	throws CertPathValidatorException, CertificateException 
    {
	validate(x509Chain,null);
    }

    /**
     * Validates a {@link X509Certificate} chain for given date.
     * @param x509Chain array of certificates, they should be in the right
     * order.
     * @param date at which to check the certificate chain, when
     * <CODE>null</CODE> use now.
     * @throws CertPathValidatorException when chain validation fails.
     * {@link Exception#getMessage()} gives the reason.
     * @throws CertificateException upon other error
     * @see #validate(X509Certificate[])
     */
    public static void validate(X509Certificate[] x509Chain, Date date)
	throws CertPathValidatorException, CertificateException 
    {
	// set date at which to check the certificate chain, when null use now
	pkixParameters.setDate(date);

	//getCertPath returns a CertPath from an array of X509Certificates
	try {
	    CertPath certPath=getCertPath(x509Chain);
	    certValidator.validate(certPath,pkixParameters);
	} catch(CertPathValidatorException e)    {
	    throw e;
	} catch(Exception e)	{
	    throw new CertificateException("Error processing certificate chain: "+e.getMessage());
	}
    }

    /**
     * Method to get all the accepted issuers for certificate chain checking.
     * @return X509Certificate[] array of accepted root signing certificates
     */
    public static X509Certificate[] getAcceptedIssuers()   {
	Vector<X509Certificate> caCerts=new Vector<X509Certificate>();
	try {
	    Enumeration<String> aliases=trustStore.aliases();
	    for (; aliases.hasMoreElements();)	{
		caCerts.add(
		    (X509Certificate)trustStore.getCertificate(aliases.nextElement())
		);
	    }
	} catch (KeyStoreException e)	{
	    return null;
	}

	// Note: issuers should be non-null, but maybe empty.
	X509Certificate[] issuers=new X509Certificate[caCerts.size()];
	caCerts.toArray(issuers);
	return issuers;
    }

    /**
     * Returns a {@link CertPath} object for the given array of {@link
     * X509Certificate}. It also strips off the root CA certificate, i.e. if the
     * first certificate is self-signed it will be stripped, this is necessary
     * to ensure that we don't get error messages from the validator if this is
     * a Version 1 cert (like many root CA certificates are).
     * @param x509Chain array of <CODE>X509Certificate</CODE> certificates, note
     * that Java demands them to be in the correct order, see
     * <A HREF="http://www.ietf.org/rfc/rfc5246.txt">RFC 5246 page 48</A>.
     * @return CertPath as constructed, if a self-signed is at the beginning of
     * the chain, it is stripped before creating the chain.
     * @throws CertPathValidatorException when creation fails
     */
    private static CertPath getCertPath(X509Certificate[] x509Chain)
	throws CertPathValidatorException
    {
	X509Certificate[] effChain=x509Chain;
	int last=x509Chain.length-1;
	X509Certificate rootCert=x509Chain[last];
	// check if first certificate is selfsigned: then strip
	X500Principal issuer=rootCert.getIssuerX500Principal();
	X500Principal subject=rootCert.getSubjectX500Principal();
	if (issuer.equals(subject)) { // Selfsigned
	    effChain=new X509Certificate[last]; // i.e. one smaller
	    System.arraycopy(x509Chain,0,effChain,0,last); // i.e. length-1
	}
	// Now build CertPath from the effChain
	try {
	    return certFactory.generateCertPath(Arrays.asList(effChain));
	} catch(CertificateException e)	{
	    throw new CertPathValidatorException(e.getMessage());
	}
    }

    /**
     * Static method creating a new {@link CertPathValidator}. Since this method
     * is intended to be called at class initialization it throws a
     * <CODE>RuntimeException</CODE> since it cannot be caught.
     * @return CertPathValidator
     * @throws RuntimeException in case of error.
     */
    private static CertPathValidator initCertValidator() 
	throws RuntimeException
    {
	// initialize BouncyCastle provider if necessary
	if (Security.getProvider("BC") == null)	{
	    try {
		Security.addProvider(new BouncyCastleProvider());
	    } catch (Exception e)   {
		throw new RuntimeException("Cannot add BouncyCastle security provider");
	    }
	}
	try {
	    // Use BouncyCastle, it gives reasonable error messages...
	    return CertPathValidator.getInstance("PKIX","BC");
	} catch(Exception e)	{
	    throw new RuntimeException("Cannot initialize CertPathValidator: "+e.getMessage());
	}
    }
    
    /**
     * Static method creating a new {@link CertificateFactory}. Since this
     * method is intended to be called at class initialization it throws a
     * <CODE>RuntimeException</CODE> since it cannot be caught.
     * @return CertificateFactory
     * @throws RuntimeException in case of error.
     */
    private static CertificateFactory initCertFactory()	throws RuntimeException
    {
	// initialize BouncyCastle provider if necessary
	if (Security.getProvider("BC") == null)	{
	    try {
		Security.addProvider(new BouncyCastleProvider());
	    } catch (Exception e)   {
		throw new RuntimeException("Cannot add BouncyCastle security provider");
	    }
	}
	try {
	    return CertificateFactory.getInstance("X.509", "BC");
	} catch(Exception e)	{
	    throw new RuntimeException("Cannot initialize CertificateFactory: "+e.getMessage());
	}
    }
    
    /**
     * Static method to initialize the {@link PKIXBuilderParameters} for {@link
     * CertPathValidator}. It uses {@link PKIXBuilderParameters} and not {@link
     * PKIXParameters} since it allows us to raise the certificate chain length
     * to be checked to infinity (-1). Since this method is intended to be
     * called at class initialization it throws a <CODE>RuntimeException</CODE>
     * since it cannot be caught.
     * @return PKIXBuilderParameters
     * @throws RuntimeException in case of error.
     */
    private static PKIXBuilderParameters initPKIXParameters() 
	throws RuntimeException
    {
	try {
	    // PKIXBuilderParameters needs X509CertSelector
	    PKIXBuilderParameters params=
		new PKIXBuilderParameters(trustStore,new X509CertSelector());
	    params.setRevocationEnabled(false);
	    // Set certchain checklength to infinite
	    params.setMaxPathLength(-1);
	    return params;
	} catch(Exception e)	{
	    throw new RuntimeException("Cannot initialize PKIXParameters: "+e.getMessage());
	}
    }
    /**
     * Method to find the correct truststore with trusted CA certificates.
     * The code for this method is taken over roughly from the Java JDK 1.6
     * internal class <CODE>sun.security.ssl.TrustManagerFactoryImpl</CODE>.
     * Since this method is intended to be called at class initialization it
     * throws a <CODE>RuntimeException</CODE> since it cannot be caught.
     * @return KeyStore containing the trusted CA certificates.
     * @throws RuntimeException in case of error.
     */
    private static KeyStore getCacertsKeyStore() throws RuntimeException
    {
        String storeFileName = null;
        File storeFile = null;
        FileInputStream fis = null;
        String defaultTrustStoreType;
        String defaultTrustStoreProvider;
        final HashMap<String,String> props = new HashMap<String,String>();
        final String sep = File.separator;
        KeyStore ks = null;

	try{ 
	    AccessController.doPrivileged(new PrivilegedExceptionAction<Void>() {
		public Void run() throws Exception {
		    props.put("trustStore", System.getProperty(
				    "javax.net.ssl.trustStore"));
		    props.put("javaHome", System.getProperty(
					    "java.home"));
		    props.put("trustStoreType", System.getProperty(
				    "javax.net.ssl.trustStoreType",
				    KeyStore.getDefaultType()));
		    // Don't use BC, since it doesn't understand JKS stores
		    props.put("trustStoreProvider", System.getProperty(
				    "javax.net.ssl.trustStoreProvider", ""));
		    props.put("trustStorePasswd", System.getProperty(
				    "javax.net.ssl.trustStorePassword", ""));
		    return null;
		}
	    });
	} catch(PrivilegedActionException e)	{
	    throw new RuntimeException("Cannot determine defaults for truststore");
	}

        /*
         * Try in the following order:
         *      javax.net.ssl.trustStore  (if this variable exists, stop)
         *      jssecacerts
         *      cacerts
         *
         * If none exists, we use an empty keystore.
         */
        storeFileName = props.get("trustStore");
        if (!"NONE".equals(storeFileName)) {
            if (storeFileName != null) {
                storeFile = new File(storeFileName);
                fis = getFileInputStream(storeFile);
            } else {
                String javaHome = props.get("javaHome");
                storeFile = new File(javaHome + sep + "lib" + sep
                                                + "security" + sep +
                                                "jssecacerts");
                if ((fis = getFileInputStream(storeFile)) == null) {
                    storeFile = new File(javaHome + sep + "lib" + sep
                                                + "security" + sep +
                                                "cacerts");
                    fis = getFileInputStream(storeFile);
                }
            }
            if (fis != null)
                storeFileName = storeFile.getPath();
            else
                storeFileName = "No File Available, using empty keystore.";
        }
        defaultTrustStoreType = props.get("trustStoreType");
        defaultTrustStoreProvider = props.get("trustStoreProvider");

        // Try to initialize trust store.
	try {
	    if (defaultTrustStoreType.length() != 0) {
		if (defaultTrustStoreProvider.length() == 0) {
		    ks = KeyStore.getInstance(defaultTrustStoreType);
		} else {
		    ks = KeyStore.getInstance(defaultTrustStoreType,
					    defaultTrustStoreProvider);
		}
		char[] passwd = null;
		String defaultTrustStorePassword = props.get("trustStorePasswd");
		if (defaultTrustStorePassword.length() != 0)
		    passwd = defaultTrustStorePassword.toCharArray();
		// if trustStore is NONE, fis will be null -> initialize
		ks.load(fis, passwd);
	    }
	} catch(Exception e)	{
	    throw new RuntimeException("Cannot load cacerts store: "+e.getMessage());
	}

	try {
	    if (fis != null)
		fis.close();
	} catch(Exception e)	{
	    throw new RuntimeException("Cannot close cacerts store: "+e.getMessage());
	}


        return ks;
    }

    /**
     * Checks whether a file exists and can be opened.
     * @param file file to be checked.
     * @return FileInputStream to the file or <CODE>null</CODE> when it could
     * not be opened or didn't exist.
     */
    private static FileInputStream getFileInputStream(final File file)	{
	try {
	    return AccessController.doPrivileged(
                new PrivilegedExceptionAction<FileInputStream>() {
                    public FileInputStream run() {
                        try {
                            if (file.exists()) {
                                return new FileInputStream(file);
                            } else {
                                return null;
                            }
                        } catch (FileNotFoundException e) {
                            return null;
                        }
                    }
                });
	} catch (PrivilegedActionException e)	{
	    // Somehow we cannot run this, hence cannot read the file either...
	    return null;
	}
    }
}

