package nl.nikhef.slcshttps.gui;

import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JLabel;
import javax.swing.JTable;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.BorderFactory;
import java.awt.Dimension;
import java.awt.Color;

import java.util.Date;

import java.security.cert.X509Certificate;
import java.security.KeyStoreException;

import nl.nikhef.slcshttps.CAHttps;

/**
 * Class to create a panel showing information about the certificate of a
 * {@link CAHttps}.
 * @author Mischa Sall&eacute;
 * @version 0.1
 * @see CAPanel
 */
public class CertPanel extends JPanel {
    /**
     * Constructs a <CODE>CertPanel</CODE>, containing a table with information
     * about the certificate in given <CODE>caHttps</CODE>.
     * @param caHttps the CA to get the certificate from.
     * @throws KeyStoreException when <CODE>caHttps</CODE> exists but
     * certificate retrieval fails.
     */
    public CertPanel(CAHttps caHttps) throws KeyStoreException	{
	// First check if there is a CA
	if (caHttps==null)  {
	    JLabel certText=new JLabel(
		"<html>CA object does not exist.<P>"+
		"Cannot show any certificate information.</html>");
	    add(certText);
	    return;
	}
	// Try getting the certificate
	X509Certificate x509Cert;
	try {
	    x509Cert=caHttps.getCertificate();
	} catch(KeyStoreException e)  {
	    throw(new KeyStoreException("CertPanel: "+e.getMessage()));
	}
	if (x509Cert==null) {
	    JLabel certText=new JLabel(
		"<html>CA object exists, but its certificate is (still) "+
		"empty.<P>Probably need to initialize the CA object or store "+
		"a certificate</html>");
	    certText.setBackground(Color.white);
	    setBackground(Color.white);
	    add(certText);
	    return;
	}

	// Setup the certificate table: certModel
        NonEditableDefaultTableModel certModel=
				    new NonEditableDefaultTableModel();
	// Headers
        certModel.setColumnIdentifiers(new String[] {
	    "Attribute", "Value" });
	// All the fields
        certModel.addRow(new String[] {
	    // toString() method works better than getName()
	    "Issuer", x509Cert.getIssuerX500Principal().toString() });
        certModel.addRow(new String[] {
	    // toString() method works better than getName()
	    "Subject", x509Cert.getSubjectX500Principal().toString() });
        certModel.addRow(new String[] {
	    "Valid from", x509Cert.getNotBefore().toString() });
	// Check the validity: for expired make it red and italic
	Date then=x509Cert.getNotAfter();
	String text=then.toString();
	String fulltext=null;
	if (then.getTime()<new Date().getTime())
	    fulltext="<HTML><FONT color=\"red\"><I>"+
		text+"</I></FONT></HTML>";
	else
	    fulltext=text;
        certModel.addRow(new String[] {
	    "Valid till", fulltext });
        certModel.addRow(new String[] {
	    "Version", Integer.toString(x509Cert.getVersion()) });
        certModel.addRow(new String[] {
	    "Signature algorithm", x509Cert.getSigAlgName() });
        certModel.addRow(new String[] {
	    "Serialnumber", 
	    CAHttps.getSerialString(x509Cert.getSerialNumber()) });

	// Create the certTable containing the certModel
        JTable certTable = new JTable(certModel);
	// It should not have focuss: no input
	certTable.setFocusable(false);

	// Change relative width of left and right columns
	certTable.getColumnModel().getColumn(0).setPreferredWidth(333);
	certTable.getColumnModel().getColumn(1).setPreferredWidth(667);
	
	// Update dimensions: width + 20%, height equal to exact table
	Dimension certDim=certTable.getPreferredScrollableViewportSize();
	certDim.width*=1.2;
	certDim.height=certTable.getRowHeight()*(certModel.getRowCount());
	certTable.setPreferredScrollableViewportSize(certDim);

	// Setup panel layout
	setLayout(new BoxLayout(this,BoxLayout.Y_AXIS));

	// Now add content to the CertPanel:
	// Header
	add(new JLabel("Certificate Information"));
	// vertical interspace
	add(Box.createRigidArea(new Dimension(0,certTable.getRowHeight()/2)));
	// A JScrollPane with the certTable
	add(new JScrollPane(certTable));

	// Setup outer border
	setBorder(BorderFactory.createCompoundBorder(
                BorderFactory.createEtchedBorder(),
                BorderFactory.createEmptyBorder(12, 12, 12, 12)));
    }
}
