package nl.nikhef.slcshttps.crypto;

import org.bouncycastle.jce.provider.BouncyCastleProvider; // BC init
import java.security.Security;		// BC init
import java.security.KeyPair;		// keypair generation
import java.security.KeyPairGenerator;	// keypair generation
import java.security.PrivateKey;	// keypair generation    
import java.security.PublicKey;		// keypair generation
import java.security.KeyStore;		// 
import java.security.Key;		// conversion for keystore
import javax.net.ssl.KeyManager;	// getKeyManager()
import javax.net.ssl.KeyManagerFactory;	// getKeyManager()
import java.util.Random;		// getPassword
import java.util.Date;			// getPassword (init random)
import java.security.cert.Certificate;	// storeCertificate()
import java.security.cert.X509Certificate;  // storeCertificate()
import java.util.Enumeration;		// importPKCS12()

import java.security.NoSuchProviderException;
import java.security.cert.CertificateException;
import java.security.KeyStoreException;
import java.security.SignatureException;

/**
 * This class is a holder for the different cryptographic objects: keypair
 * ({@link PrivateKey}/{@link PublicKey}), certificate signing request
 * ({@link CSR}) and a {@link KeyStore} containg the signed {@link
 * X509Certificate}.
 * The private key cannot be obtained directly, only indirectly by getting a
 * {@link KeyManager} for the certificate. The keypair is created at
 * construction time, or imported when using
 * {@link #importPKCS12(KeyStore,char[])}.
 * @author Mischa Sall&eacute;
 * @version 0.1
 */
public class CryptoStore    {
    /** Contains private key, will be initialized in constructor. */
    private PrivateKey privateKey;
    /** Contains public key, will be initialized in constructor. */
    private PublicKey publicKey;

    /** Contains the CSR, initialized using {@link #CSRinit()} or {@link
     * #CSRinit(String)}, can be obtained using {@link #getCSR()}. */
    private CSR csr=null;

    /** Keystore containing the signed {@link X509Certificate} and corresponding
     * {@link PrivateKey}. */
    private KeyStore keyStore=null;
    /** Password for entry in {@link #keyStore}, initialized in constructor to a
     * random value. */
    private char[] password=null;

    /** default alias for entry in {@link #keyStore} containing the privatekey
     * and certificate. */
    private static final String CERT_ALIAS="CERTIFICATE";

    /** default length for keystore-entry random {@link #password} is {@value}. */
    private static final int PASSWORD_LENGTH=16;

    /** default keylength is {@value}. */
    protected static final int KEYLENGTH=1024;

    /** 
     * Constructs a new {@link CryptoStore}, using a RSA keylength
     * <CODE>keyLen</CODE>. This initializes the {@link BouncyCastleProvider} if
     * necessary, creates the keypair, initializes the internal PKCS12 {@link
     * KeyStore} and creates the random {@link #password} with length {@link
     * #PASSWORD_LENGTH}={@value PASSWORD_LENGTH}.
     * @param keyLen the key length used for the keypair
     * @throws NoSuchProviderException if adding the {@link
     * BouncyCastleProvider} fails.
     * @throws KeyStoreException for other errors relating to key creation.
     * @see #CryptoStore()
     */
    public CryptoStore(int keyLen) throws NoSuchProviderException, KeyStoreException
    {
	/** initialize <CODE>BouncyCastle</CODE> provider if necessary */
	if (Security.getProvider("BC") == null)	{
	    try {
		Security.addProvider(new BouncyCastleProvider());
	    } catch (Exception e)   {
		throw new NoSuchProviderException("Cannot add BouncyCastle security provider");
	    }
	}

	/** initialize keypair generator and create keypair */
	try {
	    KeyPairGenerator keygen=KeyPairGenerator.getInstance("RSA");
	    keygen.initialize(keyLen);

	    KeyPair keyPair=keygen.genKeyPair();
	    privateKey=keyPair.getPrivate();
	    publicKey=keyPair.getPublic();
	} catch (Exception e)	{
	    throw new KeyStoreException("Failed to create keypair in CryptoStore",e);
	}

	/** Initialize keystore */
	try {
	    keyStore=KeyStore.getInstance("PKCS12","BC");
	    keyStore.load(null,null);
	} catch (Exception e)	{
	    throw new KeyStoreException("Failed to created keystore in CryptoStore",e);
	}

	/** Set random keystore password of PASSWORD_LENGTH characters */
	try {
	    password=getPassword(PASSWORD_LENGTH);
	} catch (KeyStoreException e)	{
	    throw new KeyStoreException("Failed to create in CryptoStore");
	}
    }

    /** 
     * Constructs a new {@link CryptoStore}, using a default RSA keylength
     * {@value KEYLENGTH}. This initializes the {@link BouncyCastleProvider} if
     * necessary, creates the keypair, initializes the internal PKCS12 {@link
     * KeyStore} and creates the random {@link #password} with length {@value
     * PASSWORD_LENGTH}.
     * @throws NoSuchProviderException if adding the {@link
     * BouncyCastleProvider} fails.
     * @throws KeyStoreException for other errors relating to key creation.
     * @see #CryptoStore(int)
     */
    public CryptoStore() throws NoSuchProviderException, KeyStoreException
    {
	this(KEYLENGTH);
    }

    /**
     * Initializes a new certificate signing Request ({@link CSR}) for the given
     * DN.
     * @param DN <CODE>String</CODE> describing the DN to use in the
     * {@link CSR}.
     * @see CSR#CSR(String,PublicKey,PrivateKey)
     * @throws SignatureException upon error.
     */
     
    public void CSRinit(String DN) throws SignatureException
    {
	try {
	    csr=new CSR(DN,publicKey,privateKey);
	} catch (Exception e)	{
	    throw new SignatureException("Cannot create a Certificate Signing Request",e);
	}
    }

    /**
     * Initializes a new certificate signing request ({@link CSR}) with a default DN.
     * @throws SignatureException
     * @see #CSRinit(String)
     * @see CSR#CSR(PublicKey,PrivateKey)
     */
    public void CSRinit() throws SignatureException
    {
	try {
	    csr=new CSR(publicKey,privateKey);
	} catch (Exception e)	{
	    throw new SignatureException("Cannot create a Certificate Signing Request",e);
	}
    }

    /** Returns the current {@link CSR}.
     * @return CSR, the current CSR.
     */
    public CSR getCSR()
    {
	return csr;
    }

    /**
     * Stores {@link X509Certificate} <CODE>x509Cert</CODE> in the internal
     * {@link #keyStore}, using the constant alias {@link #CERT_ALIAS} and
     * random password {@link #password}.
     * @param x509Cert the certificate to be stored.
     * @throws KeyStoreException upon error.
     */
    public void storeCertificate(X509Certificate x509Cert)
	throws KeyStoreException
    {
	try {
	    Certificate[] certStack=new Certificate[]{(Certificate)x509Cert};
	    keyStore.setKeyEntry(CERT_ALIAS,(Key)privateKey,password,certStack);
	} catch (Exception e)	{
	    throw new KeyStoreException("Cannot store certificate in keystore",e);
	}
    }

    /**
     * Imports a {@link X509Certificate} and keypair from an existing PKCS12
     * {@link KeyStore} protected with <CODE>password</CODE>.
     * Only the first entry in the <CODE>KeyStore</CODE> will be used and this
     * should be a KeyEntry with the certificate and private key. The public key
     * is derived from the certificate itself. The password should be the same
     * for keystore and entry inside it.
     * @param pkcs12Store PKCS12 <CODE>KeyStore</CODE>.
     * @param password PKCS12 <CODE>KeyStore</CODE> password, also password for
     * getting the private key
     * @return X509Certificate the certificate imported from the PKCS12
     * <CODE>KeyStore</CODE>.
     * @throws KeyStoreException in case the {@link KeyStore} could not be read,
     * is not PKCS12, is empty etc.
     * @throws CertificateException in case the certificate/key cannot be
     * retrieved from the {@link KeyStore}, if there is no key etc.
     * @see KeyStore#isKeyEntry(String)
     */
    public X509Certificate importPKCS12(KeyStore pkcs12Store, char[] password)
	throws CertificateException, KeyStoreException
    {
	if (pkcs12Store.getType()!="PKCS12")
	    throw new KeyStoreException("Specified KeyStore is not a PKCS12 KeyStore");
	Enumeration<String> aliases=pkcs12Store.aliases();
	if (aliases.hasMoreElements())	{
	    String alias=aliases.nextElement();
	    if (!pkcs12Store.isKeyEntry(alias)) 
		throw new CertificateException("First entry in KeyStore should be a X509Certificate/PrivateKey pair");
	    try {
		X509Certificate x509Cert=(X509Certificate)pkcs12Store.getCertificate(alias);
		publicKey=x509Cert.getPublicKey();
		privateKey=(PrivateKey)pkcs12Store.getKey(alias,password);
		storeCertificate(x509Cert);
		return x509Cert;
	    } catch(Exception e)    {
		throw new CertificateException("Cannot get certificate or keys from PKCS12 KeyStore");
	    }
	} else	{
	    throw new KeyStoreException("KeyStore is empty");
	}
    }

    /**
     * Returns the {@link X509Certificate} which was stored with given
     * alias, if this alias is for a {@link KeyStore#isKeyEntry(String)
     * KeyEntry} (which comes with a certificate chain), the first certificate
     * in the chain is returned.
     * @param alias alias for the {@link KeyStore} entry
     * @throws KeyStoreException
     * @return X509Certificate stored using the given alias.
     * @see java.security.KeyStore#getCertificate
     */
    public X509Certificate getCertificate(String alias)
	throws KeyStoreException
    {
	return (X509Certificate)keyStore.getCertificate(alias);
    }

    /**
     * Returns the {@link X509Certificate} which was stored using the default
     * {@link #CERT_ALIAS} alias, if this alias is for a {@link
     * KeyStore#isKeyEntry(String) KeyEntry} (which comes with a certificate
     * chain), the first certificate in the chain is returned.
     * @throws KeyStoreException
     * @return X509Certificate stored using the default {@link #CERT_ALIAS}.
     * @see #getCertificate(String)
     */
    public X509Certificate getCertificate() throws KeyStoreException
    {
	return getCertificate(CERT_ALIAS);
    }

    /**
     * Removes the entry in the internal {@link KeyStore} which was stored with
     * the given alias.
     * @param alias alias for the {@link KeyStore} entry
     * @throws KeyStoreException
     */
    public void deleteCertificate(String alias) throws KeyStoreException
    {
	keyStore.deleteEntry(alias);
    }

    /**
     * Removes the entry in the internal {@link KeyStore} which was stored with
     * the default alias {@link #CERT_ALIAS}.
     * @throws KeyStoreException
     * @see #deleteCertificate(String)
     */
    public void deleteCertificate()
	throws KeyStoreException
    {
	deleteCertificate(CERT_ALIAS);
    }

    /** 
     * Returns a {@link KeyManager}[] array which can be used for, for example, 
     * setting up SSL connections. Note that a <CODE>KeyManager</CODE> gives
     * public access to its private key, hence this method is package private.
     * @return KeyManager[]
     * @throws KeyStoreException in case of error
     */
    KeyManager[] getKeyManagers() throws KeyStoreException
    {
	KeyManager[] keyManagers;
	try {
	    KeyManagerFactory keyManagerFactory=KeyManagerFactory.getInstance("SunX509");
	    keyManagerFactory.init(keyStore,password);
	    keyManagers=keyManagerFactory.getKeyManagers();
	} catch(Exception e)	{
	    throw new KeyStoreException("Cannot get KeyManager[]",e);
	}
	return keyManagers;
    }

    /**
     * Creates a random password of length <CODE>length</CODE> from the set
     * [a-zA-Z0-9].
     * @param length length of password.
     * @return char[] with the password.
     * @throws KeyStoreException upon error
     */
    private char[] getPassword(int length) throws KeyStoreException
    {
	try {
	    final String chars="abcdefghijklmnopqrstuvxwyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789";
	    final int numchars=chars.length();
	    char[] password=new char[length];
	    Date date=new Date();
	    Random rnd = new Random(date.getTime());
	    for (int i=0; i<length; i++)
		password[i]=chars.charAt(rnd.nextInt(numchars));
	    date=null; rnd=null;
	} catch (Exception e)	{
	    throw new KeyStoreException("Could not create random password",e);
	}
	return password;
    }

}
