package nl.nikhef.slcshttps.crypto;

import org.bouncycastle.jce.PKCS10CertificationRequest;
import java.security.PublicKey;
import java.security.PrivateKey;

import org.bouncycastle.asn1.x509.X509Name;
import org.bouncycastle.asn1.DERSet;

import java.io.StringWriter;
import org.bouncycastle.openssl.PEMWriter;
import java.net.URLEncoder;

import java.security.MessageDigest;

import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.InvalidKeyException;
import java.security.SignatureException;
import java.io.IOException;

/**
 * This class is a holder for a PKCS10 Certificate Signing Request, which also
 * supplies methods to obtain a SHA1 hash of its DER encoding and a method to
 * obtain a URL encoded PEM string.
 * @author Mischa Sall&eacute;
 * @version 0.1
 */ 
public class CSR extends PKCS10CertificationRequest {
    /** Default subject DN certificate signing requests is {@value}. */
    protected static final String DUMMY_DN="C=X, O=Y, CN=Z";

    /** Constructs a Certificate Signing Request for the given keypair using
     * specified <CODE>subjectDN</CODE>. It uses SHA1/RSA as signature algorithm.
     * @param subjectDN specifies the subject DN to use for signing request.
     * @param pubKey specifies public key of keypair to use for signing request.
     * @param privKey specifies private key of keypair to use for signing request.
     * @throws NoSuchAlgorithmException
     * @throws NoSuchProviderException
     * @throws InvalidKeyException
     * @throws SignatureException
     */
    public CSR(String subjectDN, PublicKey pubKey, PrivateKey privKey)
	throws NoSuchAlgorithmException, NoSuchProviderException,
	       InvalidKeyException, SignatureException
    {
	super("SHA1WithRSA", new X509Name(subjectDN), pubKey, new DERSet(), privKey);
    }

    /** Constructs a Certificate Signing Request for the keypair using a default
     * {@link #DUMMY_DN} (={@value DUMMY_DN}).
     * @param pubKey specifies public key of keypair to use for signing request
     * @param privKey specifies private key of keypair to use for signing request
     * @throws NoSuchAlgorithmException
     * @throws NoSuchProviderException
     * @throws InvalidKeyException
     * @throws SignatureException
     * @see #CSR(String, PublicKey, PrivateKey)
     */
    public CSR(PublicKey pubKey, PrivateKey privKey)
	throws NoSuchAlgorithmException, NoSuchProviderException,
	       InvalidKeyException, SignatureException
    {
	this(DUMMY_DN,pubKey,privKey);
    }

    /**
     * Returns a PEMstring for the CSR as a URL encoded (UTF-8)
     * <CODE>String</CODE>.
     * @return String representing URL encoded PEMstring.
     * @throws IOException if construction of PEMstring fails
     * @see URLEncoder
     * @see PEMWriter
     */
    public String pemString() throws IOException {
	String pemencode=null;
	try {
	    StringWriter buffer=new StringWriter();
	    PEMWriter pem=new PEMWriter(buffer);
	    pem.writeObject(this);
	    pem.close();
	    pemencode=URLEncoder.encode(buffer.toString(),"UTF-8");
	} catch (Exception e)	{
	    throw new IOException("Cannot construct pemstring from CSR: "+e.getMessage());
	}
	return pemencode;
    }

    /**
     * Returns the SHA1 hash for the DER encoded CSR.
     * @return String containing the SHA1 hash for the DER encoding.
     * @throws IOException if construction of hash fails
     */
    public String hash() throws IOException {
	String dersha1=null;
	try {
	    byte[] der=this.getEncoded(); // der => DER encoded CSR
	    MessageDigest hash=MessageDigest.getInstance("SHA-1","BC"); // initialize digester
	    byte[] dig=hash.digest(der); // calculate digest
	    dersha1=bytetoString(dig);
	} catch (Exception e)	{
	    throw new IOException("Cannot construct SHA1 hash from CSR: "+e.getMessage());
	}
	return dersha1;
    }

    /**
     * returns a <CODE>String</CODE> representation of the given
     * <CODE>byte[]</CODE> array. In Java 1.5 and higher this can be done using
     * {@link String#format}<CODE>("%02x",bytes[i])</CODE>.
     * @param bytes <CODE>byte[]</CODE> array to convert
     * @return String representing the <CODE>byte[]</CODE> array
     */
    private String bytetoString(byte[] bytes) {
	// For JDK 1.5 and higher we can use:
	// String string="";
	// for (int i=0; i<bytes.length; i++)
	//      string=string.concat(String.format("%02x",bytes[i]));
	// return string;
	final String digits="0123456789abcdef";
	char[] chararray=new char[2*bytes.length];
	int x1,x2,digit;
	for (int i=0; i<bytes.length; i++)	{
	    // Need int for the shift, but have *signed* byte
	    digit=(bytes[i]>=0 ? (int)bytes[i] : (int)bytes[i]+256);
	    x1=digit>>4; x2=digit-(x1<<4);
	    chararray[2*i]=digits.charAt(x1); 
	    chararray[2*i+1]=digits.charAt(x2);
	}
	return new String(chararray);
    }
}
