package nl.nikhef.slcshttps;

import java.lang.System;
import java.io.InputStream;
import java.io.PrintStream;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.HttpsURLConnection;
import java.security.cert.X509Certificate;

import java.math.BigInteger;

import java.io.IOException;
import java.security.cert.CertificateException;
import java.security.KeyStoreException;
import java.security.SignatureException;
import java.security.KeyManagementException;

import nl.nikhef.slcshttps.crypto.CryptoStore;
import nl.nikhef.slcshttps.crypto.CryptoSSL;
import nl.nikhef.slcshttps.trust.HttxURLConnection;
import nl.nikhef.slcshttps.trust.TrustManagerImpl;

/**
 * Abstract class for communication to a Certificate Authority (CA), an
 * implementation should include methods to initialize (e.g authenticate at an
 * Online CA) and to retrieve and store a certificate. It further includes
 * methods to set and reset the default {@link SSLSocketFactory}. Setting this
 * means that it will be used for all SSL connections which will be set up
 * afterwards. All cryptographic objects such as keypairs, CSRs and certificates
 * are stored in a {@link CryptoStore} object. There is support for both {@link
 * HttpsURLConnection} and for {@link HttxURLConnection}, either independently
 * or combined. The behaviour can be driven using the property {@value
 * HTTPS_PROP}.
 *
 * @author Mischa Sall&eacute;
 * @version 0.3
 */
public abstract class CAHttps	{
    /** The default <CODE>SSLSocketFactory</CODE> for a {@link
     * HttpsURLConnection}, so that we can revert to it. */
    private static SSLSocketFactory defaultHttpsSSLSocketFactory=
	HttpsURLConnection.getDefaultSSLSocketFactory();
    /** The default <CODE>SSLSocketFactory</CODE> for a {@link
     * HttxURLConnection}, so that we can revert to it. */
    private static SSLSocketFactory defaultHttxSSLSocketFactory=
	HttxURLConnection.getDefaultSSLSocketFactory();
    /** The serial number for the currently used client side certificate in
     * {@link HttpsURLConnection}, can be retrieved using {@link
     * #getCAHttpsCertNo()}. */
    private static BigInteger CAHttpsCertNo=null;
    /** The serial number for the currently used client side certificate in
     * {@link HttxURLConnection}, can be retrieved using {@link
     * #getCAHttxCertNo()}. */
    private static BigInteger CAHttxCertNo=null;

    /** Property {@value} defines whether to use the client side certificate for
     * {@link HttxURLConnection} and/or {@link HttpsURLConnection}. Valid
     * options are <UL>
     * <LI><CODE>https</CODE> set client cert only for
     * <CODE>HttpsURLConnection</CODE>
     * <LI><CODE>httx</CODE> set client cert only for 
     * <CODE>HttxURLConnection</CODE>
     * <LI><CODE>both</CODE> set client cert for both
     * <LI><CODE>mask</CODE> set client cert for both but only show feedback
     * etc for <CODE>HttxURLConnection</CODE> (default)
     * </UL> */
    public static final String HTTPS_PROP="nl.nikhef.slcshttps.https";

    /** Is true when property {@value HTTPS_PROP} does NOT equal <CODE>httx</CODE>. */
    private static boolean useHttps;
    /** Is true when property {@value HTTPS_PROP} does NOT equal <CODE>https</CODE>. */
    private static boolean useHttx;
    /** Is true when property {@value HTTPS_PROP} equals <CODE>mask</CODE>. */
    private static boolean maskHttps;

    /** Initialize the correct internal flags depending on the value of
     * the property in {@link #HTTPS_PROP}.
     * @see #getMask()
     * @see #getHttx()
     * @see #getHttps() */
    static {
	String httpsProp=System.getProperty(HTTPS_PROP);
	// Set default:
	if (httpsProp==null)
	    httpsProp="mask";
	// Now look at the different options
	if (httpsProp.equals("https"))	{
	    useHttps=true;  useHttx=false;  maskHttps=false;
	} else if (httpsProp.equals("httx"))	{
	    useHttps=false; useHttx=true;   maskHttps=false;
	} else if (httpsProp.equals("both"))	{
	    useHttps=true;  useHttx=true;   maskHttps=false;
	} else if (httpsProp.equals("mask"))	{
	    useHttps=true;  useHttx=true;   maskHttps=true;
	}
    }

    /** Property {@value} defines whether to acknowledge successful
     * certificate import, download etc. Valid options are <CODE>true</CODE> or
     * <CODE>false</CODE>.
     * @see #getShowSuccess()
     * @see #setShowSuccess(boolean) */
    public static final String SUCCESS_PROP="nl.nikhef.slcshttps.acknowledge";

    /** Is equal to the value of {@value SUCCESS_PROP} when the latter is set,
     * or defaults to <CODE>true</CODE>. */
    static boolean showSuccess;
    static {
	String successProp=System.getProperty(SUCCESS_PROP);
	// Set default:
	if (successProp==null || !"false".equals(successProp))
	    showSuccess=true;
	else
	    showSuccess=false;
    }

    /** Contains, among others, the keypair, Certificate Signing Request
     * ({@link crypto.CSR}), certificate. Note that it needs
     * to package private, since implementing classes need to access it. */
    CryptoStore cryptoStore;

    /**************************************************************************
     * METHODS
     *************************************************************************/

    /**
     * Abstract method to initialize the communication with the CA. This could
     * be authenticating in case of an Online CA or choosing a file in the case
     * of file import.
     * @param initString <CODE>String</CODE> some string to use for
     * initialization.
     * @throws IOException upon error.
     */
    public abstract void initialize(String initString)
	throws IOException;

    /**
     * Abstract method to get certificate at CA and store it
     * in the internal {@link CryptoStore}.
     * @param storeString <CODE>String</CODE> some string to use for
     * retrieving the Certificate.
     * @throws IOException if communication goes wrong
     * @throws CertificateException if no valid certificate is returned
     * @throws KeyStoreException if the certificate cannot be stored in the
     * <CODE>CryptoStore</CODE>.
     * @throws KeyManagementException when using the certificate somehow fails.
     */
    public abstract void storeCertificate(String storeString)
	throws IOException, CertificateException, KeyStoreException,
	KeyManagementException;

    /**
     * Constructs a default <CODE>CAHttps</CODE>, this is the same as {@link
     * #CAHttps(boolean)} with the value <CODE>true</CODE>.
     * @throws KeyStoreException when initializing the internal {@link
     * CryptoStore} failed.
     * @see #CAHttps(boolean)
     */
    public CAHttps() throws KeyStoreException {
	this(null,null,null,true);
    }

    /**
     * Constructs a default <CODE>CAHttps</CODE> with or without creating a
     * Certificate Signing Request ({@link crypto.CSR}) in the internal {@link
     * CryptoStore}.
     * @param initCSR whether to initialize a Certificate Signing Request
     * ({@link crypto.CSR}) within the internal <CODE>CryptoStore</CODE>.
     * @throws KeyStoreException when initializing the internal {@link
     * CryptoStore} failed.
     */
    public CAHttps(boolean initCSR) throws KeyStoreException {
	this(null,null,null,initCSR);
    }

    /**
     * constructs a default <CODE>CAHttps</CODE> and changes
     * <CODE>stdout/stdin</CODE> to the streams specified.
     * @param myErr use this stream instead of <CODE>stderr</CODE>,
     * <CODE>null</CODE> for <CODE>stderr</CODE>
     * @param myOut use this stream instead of <CODE>stdout</CODE>,
     * <CODE>null</CODE> for <CODE>stdout</CODE>.
     * @param myIn use this stream instead of <CODE>stdin</CODE>,
     * <CODE>null</CODE> for <CODE>stdin</CODE>.
     * @throws KeyStoreException
     * @see #CAHttps()
     */
    public CAHttps(PrintStream myErr, PrintStream myOut, InputStream myIn)
	throws KeyStoreException
    {
	this(myErr,myOut,myIn,true);
    }

    /**
     * constructs a default <CODE>CAHttps</CODE>, changes
     * <CODE>stdout/stdin</CODE> to the streams specified, with or without
     * creating a Certificate Signing Request ({@link crypto.CSR}) in the
     * internal {@link CryptoStore}.
     * @param myErr use this stream instead of <CODE>stderr</CODE>,
     * <CODE>null</CODE> for <CODE>stderr</CODE>
     * @param myOut use this stream instead of <CODE>stdout</CODE>,
     * <CODE>null</CODE> for <CODE>stdout</CODE>.
     * @param myIn use this stream instead of <CODE>stdin</CODE>,
     * <CODE>null</CODE> for <CODE>stdin</CODE>.
     * @param initCSR whether to initialize a Certificate Signing Request
     * ({@link crypto.CSR}) within the internal <CODE>CryptoStore</CODE>.
     * @throws KeyStoreException
     */
    public CAHttps(PrintStream myErr, PrintStream myOut, InputStream myIn, 
		   boolean initCSR)
	throws KeyStoreException
    {
	// Optionally redirect inout/output stream
	if (myErr!=null)    { // stderr
	    try {
		System.setErr(myErr);
	    } catch (SecurityException e) {
		System.err.println("Cannot change errorstream, I/O will go to stderr");
	    }
	}
	if (myIn!=null)	{ // stdin
	    try {
		System.setIn(myIn);
	    } catch (SecurityException e) {
		System.err.println("Cannot change inputstream, I/O will come from stdin");
	    }
	}
	if (myOut!=null)    { // stdout
	    try {
		System.setOut(myOut);
	    } catch  (SecurityException e) {
		System.err.println("Cannot change outputstream, I/O will go to stdout");
	    }
	}

	// create CryptoStore object
	try {
	    cryptoStore=new CryptoStore();
	} catch (Exception e) {
	    // NoSuchProviderException for BC, KeyStoreException for CryptoStore itself
	    throw new KeyStoreException("Cannot initialize CryptoStore object: "+
					e.getMessage());
	}

	// optionally initialize CSR inside the CryptoStore
	if (initCSR)    {
	    try {
		cryptoStore.CSRinit();
	    } catch (SignatureException e)	{
		throw new KeyStoreException("Cannot create CSR in CryptoStore: "+
					    e.getMessage());
	    }
	}
    }

    /**
     * Method to get the {@link X509Certificate} currently in the
     * internal {@link CryptoStore}.
     * @return X509Certificate in the internal <CODE>CryptoStore</CODE>.
     * @throws KeyStoreException
     */
    public X509Certificate getCertificate() throws KeyStoreException	{
	X509Certificate x509Cert=null;
	try {
	    x509Cert=cryptoStore.getCertificate();
	} catch (KeyStoreException e) {
	    throw new KeyStoreException("Cannot retrieve certificate: "+
					e.getMessage());
	}
	return x509Cert;
    }

    /**
     * method to change the default {@link SSLSocketFactory} for {@link
     * HttpsURLConnection} such that it uses the certificate for client side
     * authentication.
     * @throws KeyStoreException when the initialization of the
     * {@link CryptoSSL} with the {@link CryptoStore} failed.
     * @throws KeyManagementException in case of problems setting up the default
     * <CODE>SSLSocketFactory</CODE>.
     * @see #setSSLSocketFactory()
     * @see #resetHttpsSSLSocketFactory()
     */
    public void setHttpsSSLSocketFactory() 
	throws KeyStoreException, KeyManagementException
    {
	CryptoSSL cryptoSSL;
	SSLSocketFactory sock_fac;
	// create new CryptoSSL object which is needed for getting the
	// KeyManager etc. and get a SSLSocketFactory for this KeyManager
	try {
	    cryptoSSL=new CryptoSSL(cryptoStore);
	    sock_fac=cryptoSSL.getSSLSocketFactory();
	} catch (KeyStoreException e)	{
	    throw new KeyStoreException("Cannot get needed KeyManager: "+
					e.getMessage());
	} catch (KeyManagementException e)	{
	    throw new KeyManagementException("Cannot get SSLSocketFactory: "+
					    e.getMessage());
	}
	// set the SSLSocketFactory as default for every new HttpsURLConnection.
	try {
	    HttpsURLConnection.setDefaultSSLSocketFactory(sock_fac);
	    CAHttpsCertNo=cryptoStore.getCertificate().getSerialNumber();
	} catch (Exception e) {
	    throw new KeyManagementException(
		"Cannot change default SSLSocketFactory for Https: "+
		e.getMessage());
	}
    }

    /**
     * method to restore the default {@link SSLSocketFactory} for {@link
     * HttpsURLConnection} to its startup default.
     * @throws SecurityException if permission to change is denied.
     * @throws KeyManagementException in other cases the change is not possible.
     * @see HttpsURLConnection#setDefaultSSLSocketFactory(SSLSocketFactory)
     */
    public void resetHttpsSSLSocketFactory() throws KeyManagementException    {
	try {
	    HttpsURLConnection.setDefaultSSLSocketFactory(defaultHttpsSSLSocketFactory);
	    CAHttpsCertNo=null;
	} catch (IllegalArgumentException e)	{
	    throw new KeyManagementException(
		"Cannot restore defaultHttpsSSLSocketFactory, not defined: "+
		e.getMessage());
	} catch (SecurityException e)	{
	    throw new SecurityException(
		"No permission to restore defaultHttpsSSLSocketFactory: "+
		e.getMessage());
	} catch (Exception e)	{
	    throw new KeyManagementException(
		"Error setting defaultHttpsSSLSocketFactory: "+
		e.getMessage());
	}
    }

    /**
     * method to change the default {@link SSLSocketFactory} for {@link
     * HttxURLConnection} such that it uses the certificate for client side
     * authentication.
     * @throws KeyStoreException when the initialization of the
     * {@link CryptoSSL} with the {@link CryptoStore} failed.
     * @throws KeyManagementException in case of problems setting up the default
     * <CODE>SSLSocketFactory</CODE>.
     * @see #setSSLSocketFactory()
     * @see #resetHttxSSLSocketFactory()
     */
    public void setHttxSSLSocketFactory() 
	throws KeyStoreException, KeyManagementException
    {
	CryptoSSL cryptoSSL;
	SSLSocketFactory sock_fac;
	// create new CryptoSSL object which is needed for getting the
	// KeyManager etc. and get a SSLSocketFactory for this KeyManager
	try {
	    cryptoSSL=new CryptoSSL(cryptoStore,new TrustManagerImpl());
	    sock_fac=cryptoSSL.getSSLSocketFactory();
	} catch (KeyStoreException e)	{
	    throw new KeyStoreException("Cannot get needed Key- or TrustManager: "+
					e.getMessage());
	} catch (KeyManagementException e)	{
	    throw new KeyManagementException("Cannot get SSLSocketFactory: "+
					e.getMessage());
	}
	// set the SSLSocketFactory as default for every new HttxURLConnection.
	try {
	    HttxURLConnection.setDefaultSSLSocketFactory(sock_fac);
	    X509Certificate x509Cert=cryptoStore.getCertificate();
	    HttxURLConnection.setClientExpireDate(x509Cert.getNotAfter());
	    CAHttxCertNo=x509Cert.getSerialNumber();
	} catch (Exception e) {
	    throw new KeyManagementException(
		"Cannot change default SSLSocketFactory for HttxURLConnection: "+
		e.getMessage());
	}
    }

    /**
     * method to restore the default {@link SSLSocketFactory} for {@link
     * HttxURLConnection} to its startup default.
     * @throws SecurityException if permission to change is denied.
     * @throws KeyManagementException in other cases the change is not possible.
     * @see HttxURLConnection#setDefaultSSLSocketFactory(SSLSocketFactory)
     */
    public void resetHttxSSLSocketFactory() throws KeyManagementException    {
	try {
	    HttxURLConnection.setDefaultSSLSocketFactory(defaultHttxSSLSocketFactory);
	    HttxURLConnection.setClientExpireDate(null);
	    CAHttxCertNo=null;
	} catch (IllegalArgumentException e)	{
	    throw new KeyManagementException(
		"defaultHttxSSLSocketFactory is not defined, cannot reset: "+
		e.getMessage());
	} catch (SecurityException e)	{
	    throw new SecurityException(
		"No permission to restore defaultHttxSSLSocketFactory: "+
		e.getMessage());
	} catch (Exception e)	{
	    throw new KeyManagementException(
		"Error setting defaultHttxSSLSocketFactory: "+e.getMessage());
	}
    }

    /**
     * method to change the default {@link SSLSocketFactory} for {@link
     * HttxURLConnection} and/or {@link HttpsURLConnection} such that they use
     * the certificate for client side authentication.
     * @throws KeyStoreException when the initialization of the
     * {@link CryptoSSL} with the {@link CryptoStore} failed.
     * @throws KeyManagementException in case of problems setting up the default
     * <CODE>SSLSocketFactory</CODE>.
     * @see #resetSSLSocketFactory()
     * @see #HTTPS_PROP
     */
    public void setSSLSocketFactory() throws KeyStoreException, KeyManagementException 	    {
	if (useHttps)
	    setHttpsSSLSocketFactory();
	if (useHttx)
	    setHttxSSLSocketFactory();
    }

    /**
     * method to restore the default {@link SSLSocketFactory} for {@link
     * HttpsURLConnection} and/or {@link HttpsURLConnection} to their startup
     * defaults.
     * @throws SecurityException if permission to change is denied.
     * @throws KeyManagementException in other cases the change is not possible.
     * @see #HTTPS_PROP
     */
    public void resetSSLSocketFactory() throws KeyManagementException	    {
	if (useHttps)
	    resetHttpsSSLSocketFactory();
	if (useHttx)
	    resetHttxSSLSocketFactory();
    }

    /**
     * method to return the serial number of the certificate used by the
     * {@link SSLSocketFactory} in setting up a {@link HttpsURLConnection}.
     * @return serial number of the certificate or <CODE>null</CODE> when not set.
     */
    public BigInteger getCAHttpsCertNo()    {
	return CAHttpsCertNo;
    }
    

    /**
     * method to return the serial number of the certificate used by the
     * {@link SSLSocketFactory} in setting up a {@link HttxURLConnection}.
     * @return serial number of the certificate or <CODE>null</CODE> when not set.
     */
    public BigInteger getCAHttxCertNo()    {
	return CAHttxCertNo;
    }

     /**
     * method to convert a <CODE>BigInteger</CODE> certificate serial number
     * into a <CODE>String</CODE> of the form <CODE>89:ab:12</CODE>.
     * @return <CODE>String</CODE> representation of the certificate serial
     * number, or <CODE>"none"</CODE> when not set.
     * @param serial the serial number to convert, <CODE>null</CODE> becomes
     * <CODE>"none"</CODE>
     */
    public static String getSerialString(BigInteger serial)	{
	if (serial!=null)   {
	    // First get serial as a base-16 encoded string
	    String inString=serial.toString(16);
	    // Now put in ':' signs:
	    // # of words=(length+1)/2, extra needed: (length+1)/2-1, total
	    // needed: length+(length+1)/2-1 = (length-1) + (length-1)/2 + 1 =
	    // len+len/2+1, last element=len+len/2
	    int len=inString.length()-1;
	    int chIndex=len+len/2;
	    char charArr[]=new char[chIndex+1];
	    for (int pos=0;pos<=len; pos++)	{
		charArr[chIndex--]=inString.charAt(len-pos);
		if ((pos>>1)<<1!=pos && pos<len) charArr[chIndex--]=':';
	    }
	    String outString=new String(charArr);
	    return outString;
	}
	else
	    return "none";
    }

    /**
     * Getter method for private {@link #maskHttps}.
     * @return <CODE>boolean</CODE> value of private field <CODE>maskHttps</CODE>
     * @see #HTTPS_PROP
     */
    public boolean getMaskHttps()  {
	return maskHttps;
    }

    /**
     * Getter method for private {@link #useHttps}.
     * @return <CODE>boolean</CODE> value of private field <CODE>useHttps</CODE>
     * @see #HTTPS_PROP
     */
    public boolean getUseHttps()  {
	return useHttps;
    }

    /**
     * Getter method for private {@link #useHttx}.
     * @return <CODE>boolean</CODE> value of private field <CODE>useHttx</CODE>
     * @see #HTTPS_PROP
     */
    public boolean getUseHttx()  {
	return useHttx;
    }

    /**
     * Setter method for private {@link CAHttps#showSuccess}.
     * @param show <CODE>boolean</CODE> to put into private
     * <CODE>showSuccess</CODE> field.
     * @see #SUCCESS_PROP
     * @see #getShowSuccess() */
    public static void setShowSuccess(boolean show)	{
	showSuccess=show;
    }

    /**
     * Getter method for private {@link CAHttps#showSuccess}.
     * @return <CODE>boolean</CODE> value of private field <CODE>showSuccess</CODE>
     * @see #SUCCESS_PROP
     * @see #setShowSuccess(boolean)
     */
    public static boolean getShowSuccess()	{
	return showSuccess;
    }

}
