package nl.nikhef.slcshttps;

import java.net.URL;
import javax.net.ssl.HttpsURLConnection;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.security.cert.X509Certificate;
import java.security.cert.CertificateFactory;
import java.io.BufferedReader;

import java.io.IOException;
import java.security.cert.CertificateException;
import java.security.NoSuchProviderException;

/**
 * This class provides methods to post data to a (CA) website and retrieve and
 * store the response. Its behaviour is similar to {@link HttpsURLConnection},
 * but not similar enough to make it an extension of it.
 * @author Mischa Sall&eacute;
 * @version 0.1
 * @see HttpsURLConnection
 */
public class CAConnection 	{
    /** the actual connection to the CA. */
    private HttpsURLConnection connection=null;
    /** <CODE>responseCode</CODE> is set after posting data by
     * {@link #postString(String)}. */
    private int responseCode=0;
    /** <CODE>responseMessage</CODE> is set after posting data by
     * {@link #postString(String)}. */
    private String responseMessage="";
    /** <CODE>response</CODE> is filled by {@link #storeResponse()} when {@link
     * #getCert()} fails and can be retrieved using {@link #getResponse()}. */
    private StringBuffer response=null;

    /** mime-type of HTTP POST contents is {@value}. */
    protected final String POST_CONTENT_TYPE="application/x-www-form-urlencoded";
    /** expected mime-type for a correct response type is {@value}. */
    protected final String RESP_CONTENT_TYPE="application/pkix-cert";

    /**
      * Default constructor to force handling of calling without
      * arguments.
      * @throws IOException
      * @see #CAConnection(String)
      */
    public CAConnection() throws IOException	{
	this(null);
    }

    /**
     * Constructs a <CODE>CAConnection</CODE> to the url specified by
     * <CODE>String CA_URL</CODE>.
     * @param CA_URL <CODE>String</CODE> representation of the URL to talk to.
     * @throws IOException for any error, including emptry URL.
     */
    public CAConnection(String CA_URL) throws IOException
    {
	if (CA_URL==null || CA_URL=="")	{
	    throw new IOException("Cannot open empty url");
	}
	try {
	    URL url = new URL(CA_URL);
	    connection=(HttpsURLConnection)url.openConnection();
	    connection.setInstanceFollowRedirects(false);
	    url=null;
	} catch (Exception e)	{
	    throw new IOException("Cannot open url "+CA_URL+": "+e.getMessage());
	}
    }

    /**
     * Sends <CODE>String</CODE> string to the opened URL using a HTTP POST with
     * content-type {@value POST_CONTENT_TYPE}.
     * @param string data to send
     * @return the HTTP responsecode (also stored internally) and can be
     * obtained using {@link #getResponseCode()}.
     * @throws IOException for any error.
     * @see HttpsURLConnection#getResponseCode()
     */
    public int postString(String string) throws IOException
    {
	responseCode=0; // reset response code
	OutputStream stream=null;
	OutputStreamWriter writer=null;
	// First try to set in output mode, this will also fail if the
	// connection isn't open
	try {
	    connection.setRequestProperty("Content-type",POST_CONTENT_TYPE);
	    connection.setDoOutput(true);
	    stream=connection.getOutputStream();
	} catch (Exception e)	{
	    throw new IOException("Cannot get an output connection to the CA: "+e.getMessage());
	}
	try { // Post data
	    writer = new OutputStreamWriter(stream,"UTF-8");
	    writer.write(string);
	} catch (Exception e)	{
	    throw new IOException("Failure to post data to CA: "+e.getMessage());
	} finally { // Close the writer if necessary
	    if (writer!=null) {
		writer.close(); writer=null;
	    }
	}
	try { // Now get the response code
	    responseCode=connection.getResponseCode();
	    responseMessage=connection.getResponseCode()+" "+connection.getResponseMessage();
	} catch (Exception e)	{
	    throw new IOException("Failure to get response code from CA: "+e.getMessage());
	}
	return responseCode;
    }

    /**
     * Tries to retrieve a {@link X509Certificate} from the open connection,
     * expecting content-type {@value RESP_CONTENT_TYPE}. Upon failure the
     * response is stored in {@link #response} and can be retrieved using {@link
     * #getResponse()}.
     * @return the retrieved <CODE>X509Certificate</CODE>.
     * @throws IOException if we couldn't get a response from the server.
     * @throws CertificateException if the server responded but not with a valid
     * certificate.
     */
    public X509Certificate getCert() throws CertificateException, IOException
    {
	X509Certificate x509Cert=null;
	/** First check if we have the right content-type */
	if (!connection.getHeaderField("Content-type").equals(RESP_CONTENT_TYPE))
	{
	    storeResponse();
	    throw new CertificateException("Server didn't return a valid certificate");
	}
	/** prepare certificate factory */
	try {
	    CertificateFactory certfact = CertificateFactory.getInstance("X.509","BC");
	    InputStream inputStream=connection.getInputStream();
	    x509Cert=(X509Certificate)certfact.generateCertificate(inputStream);
	} catch (NoSuchProviderException e) {
	    throw new CertificateException("Problem converting server output into X509Certificate",e); 
	} catch (CertificateException e)    {
	    throw new CertificateException("Problem converting server output into X509Certificate",e); 
	} catch (IOException e)	{
	    throw new IOException("Error reading response from server: "+e.getMessage());
	} catch (Exception e)	{ // catch all other
	    throw new IOException("Unknown error reading response from server: "+e.getMessage());
	}
	return x509Cert;
    }

    /**
     * returns the HTTP POST response code.
     * @return int representing the HTTP POST response code
     * @see #getResponseMessage()
     * @see HttpsURLConnection#getResponseCode()
     */
    public int getResponseCode()	{
	return responseCode;
    }

    /**
     * returns the HTTP POST response message, belonging to
     * <CODE>responseCode</CODE>.
     * @return String representing the HTTP POST response message
     * @see #getResponseCode()
     */
    public String getResponseMessage()	{
	return responseMessage;
    }

    /**
     * returns the CA output in <CODE>String</CODE> form, note that the output
     * is only stored when it couldn't be interpreted as {@link
     * X509Certificate}.
     * @return String representing the CA output.
     */
    public String getResponse()	{
	if (response==null)
	    return "";
	else
	    return response.toString();
    }

    /**
     * Stores the response from the server into the internal <CODE>String</CODE>
     * field {@link #response}.
     * @return number of lines read
     * @throws IOException
     */
    private int storeResponse() throws IOException
    {
	BufferedReader reader=null;
	response=new StringBuffer();
	int i=0; // Used to store number of lines
	try {
	    // Setup input stream
	    InputStream inputStream=connection.getInputStream();
	    InputStreamReader inputStreamReader=new InputStreamReader(inputStream);
	    reader = new BufferedReader(inputStreamReader);
	    String line;
	    // Store input into response
	    while ((line = reader.readLine()) != null) {
		response.append(line+"\n");
		i++;
	    }
	} catch (Exception e)	{
	    throw new IOException("Cannot store data from connection into response field: "+e.getMessage());
	} finally   { // Close reader if necessary
	    if (reader!=null)   {
		reader.close();
		reader=null;
	    }
	}
	return i;
    }
}
